
--[[
BodyTracker类：用于肢体跟踪功能
created by：caoyifan
]]


-- ClassBegin: BodyTracker
BodyTracker = class()

function BodyTracker:ctor()
	ARLOG("BodyTracker ctor")

	self._user_defined_data_handler = nil
	self._body_data = {}
	self._stomp_inspector = nil
	self._mass_center = ae.ARVec2:new_local(0.0, 0.0)
	self._data_converted = {}

end

function BodyTracker:start_body_tracking(algo_map)

	Event:addEventListener("body_tracking_data",function(data)
		self:on_receive_body_data(data)
	end)

	ARLOG("start_body_tracking")
	local msg = {}
	msg["event_name"] = "start_body_tracking"
	if (algo_map ~= nil) then
		msg["algo_map"] = algo_map
	end
	app_controller:send_message_to_native(msg)

end

function BodyTracker:stop_body_tracking()
	local msg = {}
	msg["event_name"] = "stop_body_tracking"
	app_controller:send_message_to_native(msg)
end

function BodyTracker:on_receive_body_data(data)

    if (data == nil) then
        ARLOG("BodyTracker:on_receive_body_data, data is nil")
		return
    end
	self._body_data = data["data"]

	self:handle_body_data(self._body_data)

	--如果_user_defined_data_handler为一个函数，将数据给到用户自定义的响应函数
	if (type(self._user_defined_data_handler) == 'function') then
		--note:用点获取function并执行
		self._user_defined_data_handler(data)
	end
	--如果_user_defined_data_handler为一个函数，将数据给到用户自定义的响应函数

	self:handle_specific_logic(self._body_data)
end

--设置用户自定义的数据响应函数
function BodyTracker:set_user_defined_data_handler(handler)
	self._user_defined_data_handler = handler
end

function BodyTracker:handle_body_data(data)

	if (data == nil or data[0] == nil) then
		self._data_converted = nil
		return
	end
	local first_body_data = data[0]

	self._data_converted = self:convert_data_format(first_body_data)
end

function  BodyTracker:handle_specific_logic(data)

    if (data == nil) then
    	return 
    end
    --跺脚检测
	if (self._stomp_inspector ~= nil) then
		self._stomp_inspector:inspect(self._data_converted)
	end

end

function BodyTracker:get_point_num_in_area(left_top, right_bottom)
	if (self._body_data == nil) then
        return 0
    end

	local obj = self._body_data
	local first_body_data = obj[0]

    if (first_body_data == nil) then
        return 0
    end

    if (left_top.x < right_bottom.x and left_top.y < right_bottom.y) then

        local num = 0
        for i = 0, 17, 1 do
            local data_vec3 = first_body_data[i]

			if(platform_name == "android")then
				if (data_vec3.x > left_top.x and
	                    data_vec3.y > left_top.y and
	                    data_vec3.x < right_bottom.x and
	                    data_vec3.y < right_bottom.y and
	                    data_vec3.z >= 4) then

	                num = num + 1
	            end
			else
	            if (data_vec3.x > left_top.x and
	                    data_vec3.y > left_top.y and
	                    data_vec3.x < right_bottom.x and
	                    data_vec3.y < right_bottom.y and
	                    data_vec3.z >= 4) then
	                num = num + 1
	            end
	        end
        end

        return num
    else
        ARLOG("get_point_num_in_area input error")
        return 0
    end
end


function BodyTracker:calcute_mass_center(body_data)

	local data = body_data

	local head_pos = data[0]
	local neck_pos = data[1]
	local up_body_pos = (data[2] + data[5]) * ae.ARVec2:new_local(0.5, 0.5)
	local down_body_pos = (data[8] + data[11]) * ae.ARVec2:new_local(0.5, 0.5)
	local center_body_pos = (up_body_pos + down_body_pos) * ae.ARVec2:new_local(0.5, 0.5)
	up_body_pos = (center_body_pos + up_body_pos) * ae.ARVec2:new_local(0.5, 0.5)
	down_body_pos = (center_body_pos + down_body_pos) * ae.ARVec2:new_local(0.5, 0.5)

	local left_upper_arm_pos = (data[5] + data[6]) * ae.ARVec2:new_local(0.5, 0.5)
	local left_forearm_pos = (data[6] + data[7]) * ae.ARVec2:new_local(0.5, 0.5)
	local left_hand_pos = data[7]

	local right_upper_arm_pos = (data[2] + data[3]) * ae.ARVec2:new_local(0.5, 0.5)
	local right_forearm_pos = (data[3] + data[4]) * ae.ARVec2:new_local(0.5, 0.5)
	local right_hand_pos = data[4]

	local left_big_leg_pos = (data[11] + data[12]) * ae.ARVec2:new_local(0.5, 0.5)
	local left_small_leg_pos = (data[12] + data[13]) * ae.ARVec2:new_local(0.5, 0.5)
	local left_foot_pos = data[13]

	local right_big_leg_pos = (data[8] + data[9]) * ae.ARVec2:new_local(0.5, 0.5)
	local right_small_leg_pos = (data[9] + data[10]) * ae.ARVec2:new_local(0.5, 0.5)
	local right_foot_pos = data[10]

	local center_pos = head_pos * ae.ARVec2:new_local(0.0862, 0.0862) +
			up_body_pos * ae.ARVec2:new_local(0.1682, 0.1682) +
			down_body_pos * ae.ARVec2:new_local(0.2793, 0.2793) +
			left_upper_arm_pos * ae.ARVec2:new_local(0.0243, 0.0243) +
			right_upper_arm_pos * ae.ARVec2:new_local(0.0243, 0.0243) +
			left_forearm_pos * ae.ARVec2:new_local(0.0125, 0.0125) +
			right_forearm_pos * ae.ARVec2:new_local(0.0125, 0.0125) +
			left_hand_pos * ae.ARVec2:new_local(0.0064, 0.0064) +
			right_hand_pos * ae.ARVec2:new_local(0.0064, 0.0064) +
			left_big_leg_pos * ae.ARVec2:new_local(0.1419, 0.1419) +
			right_big_leg_pos * ae.ARVec2:new_local(0.1419, 0.1419) +
			left_small_leg_pos * ae.ARVec2:new_local(0.0367, 0.0367) +
			right_small_leg_pos * ae.ARVec2:new_local(0.0367, 0.0367) +
			left_foot_pos * ae.ARVec2:new_local(0.0148, 0.0148) +
			right_foot_pos * ae.ARVec2:new_local(0.0148, 0.0148)

	self._mass_center = center_pos
end

function BodyTracker:convert_data_format(body_data)
	local result = {}
	for i = 0, 17, 1 do
		local data_vec3 = body_data[i]

		local pos = ae.ARVec2:new_local(data_vec3.x, data_vec3.y)
		result[i] = pos
	end

	return result

end

function BodyTracker:get_mass_center_screen_pos()
    if (self._data_converted == nil) then
        return ae.ARVec2:new_local(0,0)
    end
    self:calcute_mass_center(self._data_converted)
	return self._mass_center
end

function BodyTracker:set_stomp_trigger(trigger)
	self._stomp_inspector = StompInspector.new()
	self._stomp_inspector:set_stomp_trigger(trigger)
end

-- ClassEnd: BodyTracker


--[[
StompInspector:用于检查肢体是否在进行跺脚操作，是一个很具体的类，不具有参考意义
]]
-- ClassBegin: StompInspector
StompInspector = class()

function StompInspector:ctor()
	ARLOG("StompInspector ctor")

	self._trigger = nil
	self._left_foot_up = false
	self._left_foot_down = false
	self._left_lifting_count = 0
	self._left_falling_count = 0

	self._right_foot_up = false
	self._right_foot_down = false
	self._right_lifting_count = 0
	self._right_falling_count = 0

	self._left_foot_value_quene = ARUtils.Quene.new("left_foot_value_quene", 5)
	self._right_foot_value_quene = ARUtils.Quene.new("right_foot_value_quene", 5)
	self._length_body = 0
	self._distance_y_foots = 0

	self._left_foot_pos = ae.ARVec3:new_local(0,0,0)
	self._right_foot_pos = ae.ARVec3:new_local(0,0,0)

	--插值长度, 取前三帧数与当前帧进行抬脚跺脚的判断
	self._interpolation_buffer_size = 4

	self._data = nil
end

function StompInspector:inspect(data)

	--ARLOG("inspect")
	if (data == nil) then
		return
	end

	self:collect_relative_data(data)

	if (self:check_total_deviation(data) == "large_deviation") then
		self:restart_foot_check()
	else
		self:left_foot()
		self:right_foot()
	end
	--ARLOG("left :"..tostring(self._left_lifting_count).." "..tostring(self._left_falling_count))

	--ARLOG("right :"..tostring(self._right_lifting_count).." "..tostring(self._right_falling_count))
end

function StompInspector:collect_relative_data(data)
	if (self._left_foot_value_quene:size() < self._interpolation_buffer_size) then
		local foot_value_y = data[13].y
		self._left_foot_value_quene:push(foot_value_y)
	else
		self._left_foot_value_quene:pop()
		local foot_value_y = data[13].y
		self._left_foot_value_quene:push(foot_value_y)
	end

	if (self._right_foot_value_quene:size() < self._interpolation_buffer_size) then
		local foot_value_y = data[10].y
		self._right_foot_value_quene:push(foot_value_y)
	else
		self._right_foot_value_quene:pop()
		local foot_value_y = data[10].y
		self._right_foot_value_quene:push(foot_value_y)
	end

	local pos_r_shouler = data[2]
	local pos_l_hip = data[11]
	local length1 = self:calculate_2d_length(pos_r_shouler, pos_l_hip)

	local pos_l_shouler = data[5]
	local pos_r_hip = data[8]
	local length2 = self:calculate_2d_length(pos_l_shouler, pos_r_hip)
	self._length_body = (length1 + length2) / 2.0

	self._left_foot_pos = data[13]
	self._right_foot_pos = data[10]

	--ARLOG("left foot pos:"..self._left_foot_pos:to_string())
	--ARLOG("right foot pos:"..self._right_foot_pos:to_string())

	self._distance_y_foots = math.abs(self._right_foot_pos.y - self._left_foot_pos.y)

end

function StompInspector:check_total_deviation(data)
	if (self._data == nil) then
		self._data = data
		return
	end

	local total_delta = 0.0

	for i = 0, 17, 1 do
		local delta_vec2 = self._data[i] - data[i]
		local length = delta_vec2:length()
		local coeffient = self:get_partly_coeffient(i)
		total_delta = total_delta + coeffient * length
	end

	self._data = data

	--ARLOG("total_delta:"..tostring(total_delta))

	if (total_delta < 0.01) then
		return "little_deviation"
	elseif (total_delta < 0.04) then
		return "middle_deviation"
	else
		return "large_deviation"
	end

end

function StompInspector:get_partly_coeffient(index)
	--head
	if (index == 0) then return 0.01724 end
	if (index == 14) then return 0.01724 end
	if (index == 15) then return 0.01724 end
	if (index == 16) then return 0.01724 end
	if (index == 17) then return 0.01724 end


	--up_body
	if (index == 1) then return 0.158 end
	if (index == 2) then return 0.158 end
	if (index == 5) then return 0.158 end

	--down_body
	if (index == 8) then return 0.12 end
	if (index == 11) then return 0.12 end

	--right_arm
	if (index == 3) then return 0.0243 end
	if (index == 4) then return 0.0064 end

	--left_arm
	if (index == 6) then return 0.0243 end
	if (index == 7) then return 0.0064 end

	--right_leg
	if (index == 9) then return 0.0419 end
	if (index == 10) then return 0.0232 end

	--left_leg
	if (index == 12) then return 0.0419 end
	if (index == 13) then return 0.0232 end

end


function StompInspector:left_foot()

	local left_status = self:foot_status("check_left", self._left_foot_value_quene)

	if (left_status == "lifting") then
		self._left_lifting_count = self._left_lifting_count + 1

		if (self._left_lifting_count >= 2) then
			self._left_foot_up = true
		end
	end

	if (left_status == "falling") then
		self._left_falling_count = self._left_falling_count + 1

		if (self._left_falling_count >= 2 and self._left_foot_up) then
			self._left_foot_down = true
		end
	end

	if (self._left_foot_down) then
		if (self._distance_y_foots < self._length_body * 0.15) then
			self._trigger("left_foot", self._left_foot_pos)
			self._left_foot_up = false
			self._left_foot_down = false
			self._left_lifting_count = 0
			self._left_falling_count = 0
		end
	end
end

function StompInspector:right_foot()

	local right_status = self:foot_status("check_right", self._right_foot_value_quene)

	if (right_status == "lifting") then
		self._right_lifting_count = self._right_lifting_count + 1

		if (self._right_lifting_count >= 2) then
			self._right_foot_up = true
		end
	end

	if (right_status == "falling") then
		self._right_falling_count = self._right_falling_count + 1

		if (self._right_falling_count >= 2 and self._right_foot_up) then
			self._right_foot_down = true
		end
	end

	if (self._right_foot_down) then
		if (self._distance_y_foots < self._length_body * 0.15) then
			self._trigger("right_foot", self._right_foot_pos)
			self._right_foot_up = false
			self._right_foot_down = false
			self._right_lifting_count = 0
			self._right_falling_count = 0
		end
	end

end

function StompInspector:restart_foot_check()
	self._right_foot_up = false
	self._right_foot_down = false
	self._right_lifting_count = 0
	self._right_falling_count = 0

	self._left_foot_up = false
	self._left_foot_down = false
	self._left_lifting_count = 0
	self._left_falling_count = 0

end

function StompInspector:foot_status(tag, foot_quene)
	if (foot_quene:size() == self._interpolation_buffer_size) then
		local array = foot_quene:get_array()

		--前三帧与前两帧的数据
		local value1 = array[1] + array[2]
		--前一帧与当前帧的数据
		local value2 = array[3] + array[4]

		value1 = value1 * 2 / self._interpolation_buffer_size
		value2 = value2 * 2 / self._interpolation_buffer_size

		local buffer = self._length_body / 10

		--表示之前的数据在下面
		if (value1 > value2) then
			local gap = value1 - value2
			if (gap > buffer) then
				return "lifting"
			else
				return "lifting_little"
			end
		end

		--之前的数据在上面
		if (value2 > value1) then
			local gap = value2 - value1
			if (gap > buffer) then
				return "falling"
			else
				return "falling_little"
			end
		end

		return "unkonwn_error"

	end

	return "quene_size_error"

end

function StompInspector:calculate_2d_length(point1, point2)
	local result = 0
	local v1 = point1 - point2
	result = math.pow(v1.x, 2) + math.pow(v1.y, 2)
	result = math.sqrt(result)
	return result
end

function StompInspector:set_stomp_trigger(trigger)
	if (type(trigger) == 'function') then
		self._trigger = trigger
	else 
		ARLOG("stompInspector:set_stomp_trigger:trigger is not a function ")
	end
end

-- ClassEnd: StompInspector


