#import "RTCBoardCallView.h"
#import <sys/utsname.h>
#import <WebKit/WebKit.h>
#import <BaiduRtcBoard/BaiduRtcBoardApi.h>
#import "RTCSettingsModel.h"

static NSArray<NSString *> *kBoardToolsTable() {
    return  @[@"pen", @"arrow", @"line", @"rectangle", @"diamond", @"ellipse"];
}

static NSArray<UIColor *> *kBoardColorTable() {
    return @[
        [UIColor blackColor],
        [UIColor lightGrayColor],
        [UIColor redColor],
        [UIColor orangeColor],
        UIColorFromRGBA(0xEEAD0E, 1.0),
        UIColorFromRGBA(0x71C671, 1.0),
        UIColorFromRGBA(0x4EEE94, 1.0),
        [UIColor blueColor],
        [UIColor purpleColor],
    ];
}

static NSArray<UIColor *> *kBackgroundColorTable() {
    return @[
        UIColorFromRGBA(0xF7F7F7, 1.0),
        [UIColor blackColor],
        [UIColor orangeColor],
        UIColorFromRGBA(0x71C671, 1.0),
        [UIColor blueColor],
        [UIColor purpleColor],
    ];
}

static int kBoardToolsType[6] = {
    RTC_BOARD_TOOL_TYPE_PEN,
    RTC_BOARD_TOOL_TYPE_ARROW,
    RTC_BOARD_TOOL_TYPE_LINE,
    RTC_BOARD_TOOL_TYPE_RECT,
    RTC_BOARD_TOOL_TYPE_DIAMOND,
    RTC_BOARD_TOOL_TYPE_OVAL,
};

@interface RTCBoardCallView () <RTCRadioButtonDelegate, RTCBoardSettingDelegate>

@property (nonatomic, strong) RTCBoardView *boardView;

@property (nonatomic, strong) RTCBoardParam *param;

@end

@implementation RTCBoardCallView {
    // header view
    UIView *_headerView;
    UIButton *_undoButton;
    UIButton *_redoButton;
    UIButton *_boardButton;
    UIButton *_boardDelButton;
    UIButton *_closeButton;
    UILabel* _titleLabel;

    // page view
    UIButton *_prePageButton;
    UIButton *_nextPageButton;
    UILabel* _pageLabel;
    UIView *_pageView;

    // bottom view
    UIView *_bottomView;
    UIButton *_selectButton;
    UILabel* _selectLabel;
    UIButton *_penButton;
    UILabel* _penLabel;
    UIButton *_textButton;
    UILabel* _textLabel;
    UIButton *_eraserButton;
    UILabel* _eraserLabel;
    UIButton *_clearButton;
    UILabel* _clearLabel;
    UIButton *_moreButton;
    UILabel* _moreLabel;

    UIView *_colorView;
    UIView *_toolsView;
    // setting views
    UIView *_moreView;

    // background transparency
    int _bgTransparency;
    // tools type index
    int _toolsType;
    // tools color index
    int _toolsColor;
    // background color index
    int _backgroundColor;
    // avoid frequently request webview URL
    BOOL _requestURL;
    // current page index
    int _pageIndex;
    // total page count
    int _totalPageCount;
    // audience role
    BOOL _audience;
}

@synthesize delegate = _delegate;

- (instancetype)initWithFrame:(CGRect)frame param:(RTCBoardParam *)param {
    if (self = [super initWithFrame:frame]) {
        _requestURL = NO;
        _bgTransparency = [[[RTCSettingsModel sharedInstance] currentBoardTransparencyFromStore] intValue];

        _toolsType = [[[RTCSettingsModel sharedInstance] currentBoardToolTypeFromStore] intValue];
        _toolsColor = [[[RTCSettingsModel sharedInstance] currentBoardToolColorFromStore] intValue];
        _audience = [[RTCSettingsModel sharedInstance] currentBoardAudienceFromStore];
        _backgroundColor = [[[RTCSettingsModel sharedInstance] currentBoardBgColorFromStore] intValue];
        [self updateBackground];

        _pageIndex = 0;
        _totalPageCount =1;
        self.param = param;
        // header views
        [self addHeaderViews];
        // boardView
        [self addSubview:self.boardView];
        // page views
        [self addPageViews];
        // bottom views
        [self addBottomViews];

        CGFloat footY = SCREEN_HEIGHT * 2003 * kLayoutHeightScale;
        // select color radio button, default hidden
        CGFloat boardX = 6;
        CGRect colorFrame = CGRectMake(0, (footY - 60), SCREEN_WIDTH, 60 - 1);
        _colorView = [[RTCRadioButtonViews alloc] initWithFrame:colorFrame hasBorder:YES 
                                                           type:RTC_RADIO_BUTTON_TYPE_COLOR
                                                      viewWidth:24 viewCount:9 index:_toolsColor delegate:self];
        [self addSubview:_colorView];

        // select tools
        CGRect toolsFrame = CGRectMake(0, (footY - 60 - 70), SCREEN_WIDTH, 70);
        _toolsView = [[RTCRadioButtonViews alloc] initWithFrame:toolsFrame hasBorder:YES 
                        type:RTC_RADIO_BUTTON_TYPE_TOOLS viewWidth:30 viewCount:6 index:_toolsType delegate:self];
        [self addSubview:_toolsView];

        // more settings, including background transpancy, show stats, snapshot etc
        CGRect moreFrame = CGRectMake(0, (footY - 240), SCREEN_WIDTH, 240 - 1);
        _moreView = [[RTCBoardSettingView alloc] initWithFrame:moreFrame transparency:_bgTransparency colorIndex:_backgroundColor delegate:self];
        [self addSubview:_moreView];
    }
    return self;
}

- (void)dealloc {
    self.boardView = NULL;
    [[BaiduRtcBoardApi sharedManager] destroyBoardView];
    NSLog(@"[wb] RTCBoardCallView dealloc~");
}

- (void)onRadioButtonAdd:(RTCRadioButtonType)type button:(UIButton *)button index:(int)index {
    if (type == RTC_RADIO_BUTTON_TYPE_COLOR) {
        button.backgroundColor = kBoardColorTable()[index];
    } else if (type == RTC_RADIO_BUTTON_TYPE_TOOLS) {
        UIImage *image = [UIImage imageNamed:kBoardToolsTable()[index]];
        [button setImage:image forState:UIControlStateNormal];
    }
}

- (void)onRadioButtonSelect:(RTCRadioButtonType)type button:(UIButton *)button index:(int)index {
    if (type == RTC_RADIO_BUTTON_TYPE_COLOR) {
        [[BaiduRtcBoardApi sharedManager] setBrushColor:kBoardColorTable()[index]];
        [[RTCSettingsModel sharedInstance] storeBoardToolColor:@(index)];
    } else if (type == RTC_RADIO_BUTTON_TYPE_TOOLS) {
        [[BaiduRtcBoardApi sharedManager] setToolType:kBoardToolsType[index]];
        [[RTCSettingsModel sharedInstance] storeBoardToolType:@(index)];
    }
}

- (void)onBackgroundTransparencyChanged:(int)value {
    _bgTransparency = value;
    [[RTCSettingsModel sharedInstance] storeBoardTransparency:@(value)];
    [self updateBackground];
}

- (void)onBackgroundColorChanged:(int)index {
    _backgroundColor = index;
    [[RTCSettingsModel sharedInstance] storeBoardBgColor:@(index)];
    [self updateBackground];
}

- (void)onBoardCreateSucceed {
}

- (void)updateBackground {
    UIColor *color = kBackgroundColorTable()[_backgroundColor];
    self.backgroundColor = [color colorWithAlphaComponent:_bgTransparency / 100.0f];
    // [[BaiduRtcBoardApi sharedManager] setBackgroundColor:kBackgroundColorTable()[index] transparency:_bgTransparency / 100.0f];
}

- (void)onSnapshotTaken {
    if (self.delegate && [self.delegate respondsToSelector:@selector(onSnapshotTaken)]) {
        [self.delegate onSnapshotTaken];
    }
}

- (void)addHeaderViews {
    _headerView = [[UIView alloc] initWithFrame:CGRectZero];
    [self addSubview:_headerView];

    _closeButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addLButton:_closeButton title:@"Hide" enable:YES action:@selector(onTitleAction:)];

    _undoButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_undoButton imageName:@"pre1.png" enable:NO action:@selector(onTitleAction:)];

    _redoButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_redoButton imageName:@"next1.png" enable:NO action:@selector(onTitleAction:)];

    _boardButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_boardButton imageName:@"board.png" enable:!_audience action:@selector(onTitleAction:)];

    _boardDelButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_boardDelButton imageName:@"board1.png" enable:!_audience action:@selector(onTitleAction:)];

    _titleLabel = [[UILabel alloc] initWithFrame:CGRectZero];
    [self addFLabel:_titleLabel title:@"Whiteboard" size:16]; // param.channelName
}

- (void)addPageViews {
    _pageView = [[UIView alloc] initWithFrame:CGRectZero];
    [self addSubview:_pageView];

    _prePageButton = [self addXButtonWith:_pageView imageName:@"pagepre.png" enable:NO action:@selector(onPageAction:)];

    _nextPageButton = [self addXButtonWith:_pageView imageName:@"pagenext.png" enable:NO action:@selector(onPageAction:)];

    _pageLabel = [self addXLabelWith:_pageView title:@"1/1"];
}

- (void)addBottomViews {
    _bottomView = [[UIView alloc] initWithFrame:CGRectZero];
    [self addSubview:_bottomView];

    _selectButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_selectButton imageName:@"select.png" enable:!_audience action:@selector(onToolsAction:)];
    _selectLabel = [[UILabel alloc] initWithFrame:CGRectZero];
    [self addFLabel:_selectLabel title:@"Select" size:12];

    _penButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_penButton imageName:@"pen.png" enable:!_audience action:@selector(onToolsAction:)];
    _penLabel = [[UILabel alloc] initWithFrame:CGRectZero];
    [self addFLabel:_penLabel title:@"Pen" size:12];

    _textButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_textButton imageName:@"text.png" enable:!_audience action:@selector(onToolsAction:)];
    _textLabel = [[UILabel alloc] initWithFrame:CGRectZero];
    [self addFLabel:_textLabel title:@"Text" size:12];

    _eraserButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_eraserButton imageName:@"eraser.png" enable:!_audience action:@selector(onToolsAction:)];
    _eraserLabel = [[UILabel alloc] initWithFrame:CGRectZero];
    [self addFLabel:_eraserLabel title:@"Eraser" size:12];

    _clearButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_clearButton imageName:@"clear.png" enable:!_audience action:@selector(onToolsAction:)];
    _clearLabel = [[UILabel alloc] initWithFrame:CGRectZero];
    [self addFLabel:_clearLabel title:@"Clear" size:12];

    _moreButton = [UIButton buttonWithType:UIButtonTypeCustom];
    [self addXButton:_moreButton imageName:@"more.png" enable:YES action:@selector(onToolsAction:)];
    _moreLabel = [[UILabel alloc] initWithFrame:CGRectZero];
    [self addFLabel:_moreLabel title:@"More" size:12];
}

- (void)addLButton:(UIButton *)button title:(NSString *)title enable:(BOOL)enable action:(SEL)action {
    button.layer.masksToBounds = YES;
    button.enabled = enable;
    [button setTitle:title forState:UIControlStateNormal];
    [button setTitleColor:[UIColor redColor] forState:UIControlStateNormal];
    button.titleLabel.font = [UIFont fontWithName:@"PingFangSC-Regular" size:14];
    [button addTarget:self action:action forControlEvents:UIControlEventTouchUpInside];
    [self addSubview:button];
}

// 
- (void)addXButton:(UIButton *)button imageName:(NSString *)imageName enable:(BOOL)enable action:(SEL)action {
    button.layer.masksToBounds = YES;
    button.enabled = enable;
    UIImage *image = [UIImage imageNamed:imageName];
    [button setImage:image forState:UIControlStateNormal];
    [button addTarget:self action:action forControlEvents:UIControlEventTouchUpInside];
    [self addSubview:button];
}

- (UIButton *)addXButtonWith:(UIView *)view imageName:(NSString *)imageName enable:(BOOL)enable action:(SEL)action {
    UIButton *button = [UIButton buttonWithType:UIButtonTypeCustom];
    button.layer.masksToBounds = YES;
    button.enabled = enable && !_audience;
    UIImage *image = [UIImage imageNamed:imageName];
    [button setImage:image forState:UIControlStateNormal];
    [button addTarget:self action:action forControlEvents:UIControlEventTouchUpInside];
    [view addSubview:button];
    return button;
}

- (void)addXLabel:(UILabel *)label title:(NSString *)title {
    label.adjustsFontSizeToFitWidth = YES;
    label.textColor = [UIColor darkGrayColor];
    label.text = title;
    [self addSubview:label];
}

- (UILabel *)addXLabelWith:(UIView *)view title:(NSString *)title {
    UILabel *label = [[UILabel alloc] initWithFrame:CGRectZero];
    label.adjustsFontSizeToFitWidth = YES;
    label.textColor = [UIColor darkGrayColor];
    label.text = title;
    [view addSubview:label];
    return label;
}

- (void)addFLabel:(UILabel *)label title:(NSString *)title size:(int)size {
    label.font = [UIFont fontWithName:@"PingFangSC-Regular" size:size];
    label.textColor = [UIColor darkGrayColor];
    label.text = title;
    [self addSubview:label];
}

- (void)layoutXButton:(UIButton *)button offsetX:(CGFloat)offsetX offsetY:(CGFloat)offsetY buttonW:(CGFloat)buttonW {
    CGFloat buttonH = buttonW;
    CGFloat buttonX = offsetX;
    CGFloat buttonY = offsetY;
    CGRect buttonFrame = CGRectMake(buttonX, buttonY, buttonW, buttonH);
    button.layer.cornerRadius = buttonW / 2.0;
    button.frame = buttonFrame;
}

- (void)layoutSubviews {
    // header view
    CGFloat buttonW = SCREEN_WIDTH * 100 * kLayoutWidthScale;
    CGFloat buttonX = 10;
    CGFloat buttonY = SCREEN_HEIGHT * 124 * kLayoutHeightScale;
    CGFloat boardY = SCREEN_HEIGHT * 220 * kLayoutHeightScale;
    _headerView.layer.opaque = true;
    _headerView.layer.backgroundColor = [UIColor whiteColor].CGColor;
    _headerView.frame = CGRectMake(0, 0, SCREEN_WIDTH, boardY);

    [self layoutXButton:_undoButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];

    buttonX += (buttonW + 10);
    [self layoutXButton:_redoButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];

    buttonX += (buttonW + 10);
    [self layoutXButton:_boardButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];

    buttonX += (buttonW + 10);
    [self layoutXButton:_boardDelButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];
    _boardDelButton.hidden = YES;

    _closeButton.frame = CGRectMake(SCREEN_WIDTH - buttonW * 3 / 2, buttonY, buttonW * 3 / 2, buttonW);
    buttonX += (buttonW + 10);
    _titleLabel.frame = CGRectMake(buttonX, buttonY, buttonW * 4, buttonW);

    // page
    CGFloat pageX = 10;
    CGFloat pageY = SCREEN_HEIGHT * 1760 * kLayoutHeightScale;
    _pageView.layer.opaque = true;
    _pageView.layer.backgroundColor = [UIColor whiteColor].CGColor;
    _pageView.layer.borderWidth = 1;
    _pageView.layer.borderColor = UIColorFromRGBA(0xD3D3D3, 1.0).CGColor;
    _pageView.frame = CGRectMake(pageX, pageY, buttonW * 2 + 10, buttonW);
    _pageView.hidden = YES;

    [self layoutXButton:_prePageButton offsetX:0 offsetY:0 buttonW:buttonW];
    _pageLabel.frame = CGRectMake(buttonW - 5, 0, buttonW / 2, buttonW);
    [self layoutXButton:_nextPageButton offsetX:buttonW + 10 offsetY:0 buttonW:buttonW];

    // boardView
    CGRect  boardFrame = CGRectMake(0, 0, 0, 0);
    CGFloat boardX = SCREEN_WIDTH * 0 * kLayoutWidthScale;
    CGFloat boardW = SCREEN_WIDTH * 1260 * kLayoutWidthScale;
    CGFloat boardH = SCREEN_HEIGHT * 1783 * kLayoutHeightScale;
    boardFrame = CGRectMake(boardX, boardY, boardW, boardH);
    self.boardView.frame = boardFrame;

    // buttom view
    buttonX = 10;
    buttonY = SCREEN_HEIGHT * 1980 * kLayoutHeightScale;
    buttonW = SCREEN_WIDTH * 170 * kLayoutWidthScale;

    CGFloat bottomViewY =  boardY + boardH;
    _bottomView.layer.opaque = true;
    _bottomView.layer.backgroundColor = [UIColor whiteColor].CGColor;
    _bottomView.frame = CGRectMake(0, bottomViewY, SCREEN_WIDTH, SCREEN_HEIGHT - bottomViewY);

    [self layoutXButton:_selectButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];
    CGFloat labelY = SCREEN_HEIGHT * 2060 * kLayoutHeightScale;
    CGFloat labelW = buttonW * 5 / 8;
    _selectLabel.frame = CGRectMake(buttonX + 12, labelY, labelW, labelW);

    buttonX += (buttonW + 10);
    [self layoutXButton:_penButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];
    _penLabel.frame = CGRectMake(buttonX + 18, labelY, labelW, labelW);

    buttonX += (buttonW + 10);
    [self layoutXButton:_textButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];
    _textLabel.frame = CGRectMake(buttonX + 16, labelY, labelW, labelW);

    buttonX += (buttonW + 10);
    CGFloat labelE = buttonW * 6 / 8;
    [self layoutXButton:_eraserButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];
    _eraserLabel.frame = CGRectMake(buttonX + 10, labelY, labelE, labelW);

    buttonX += (buttonW + 10);
    [self layoutXButton:_clearButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];
    _clearLabel.frame = CGRectMake(buttonX + 14, labelY, labelW, labelW);

    buttonX += (buttonW + 10);
    [self layoutXButton:_moreButton offsetX:buttonX offsetY:buttonY buttonW:buttonW];
    _moreLabel.frame = CGRectMake(buttonX + 14, labelY, labelW, labelW);

    if (!_requestURL) {
        _requestURL = YES;
        [_boardView loadRequest:[NSURLRequest requestWithURL:[NSURL URLWithString:self.param.url]]];
    }
}

-(RTCBoardView *)boardView {
    if (!_boardView) {
        _boardView = [[BaiduRtcBoardApi sharedManager] createBoardViewWithFrame:CGRectZero];
    }
    return _boardView;
}

- (void)onTitleAction:(id)sender {
    if (sender == _closeButton) {
        if (self.delegate && [self.delegate respondsToSelector:@selector(onActionLeave)]) {
            [self.delegate onActionLeave];
        }
    } else if (sender == _undoButton) {
        [[BaiduRtcBoardApi sharedManager] undo];
    } else if (sender == _redoButton) {
        [[BaiduRtcBoardApi sharedManager] redo];
    } else if (sender == _boardButton) {
        [[BaiduRtcBoardApi sharedManager] addBoard];
        _pageView.hidden = NO;
        _boardDelButton.hidden = NO;
    } else if (sender == _boardDelButton) {
        if (_totalPageCount > 1) {
            _totalPageCount -= 1;
            [[BaiduRtcBoardApi sharedManager] deleteBoard:_pageIndex];
        }
    }
}

- (void)hidePenMenuViews {
    _colorView.hidden = YES;
    _toolsView.hidden = YES;
}

- (void)hideSettingMenuViews {
    _moreView.hidden = YES;
}

- (void)hideMenuViews {
    [self hideSettingMenuViews];
    [self hidePenMenuViews];
}

- (void)onToolsAction:(id)sender {
    if (sender == _selectButton) {
        [self hideMenuViews];
        [[BaiduRtcBoardApi sharedManager] setToolType:RTC_BOARD_TOOL_TYPE_MOUSE];
    } else if (sender == _penButton) {
        [self hideSettingMenuViews];
        _colorView.hidden = !_colorView.hidden;
        _toolsView.hidden = !_toolsView.hidden;
        _toolsType = [[[RTCSettingsModel sharedInstance] currentBoardToolTypeFromStore] intValue];
        _toolsColor = [[[RTCSettingsModel sharedInstance] currentBoardToolColorFromStore] intValue];
        [[BaiduRtcBoardApi sharedManager] setToolType:kBoardToolsType[_toolsType]];
        [[BaiduRtcBoardApi sharedManager] setBrushColor:kBoardColorTable()[_toolsColor]];
    } else if (sender == _textButton) {
        [self hideMenuViews];
        [[BaiduRtcBoardApi sharedManager] setToolType:RTC_BOARD_TOOL_TYPE_TEXT];
    } else if (sender == _eraserButton) {
        [self hideMenuViews];
        [[BaiduRtcBoardApi sharedManager] setToolType:RTC_BOARD_TOOL_TYPE_ERASER];
    } else if (sender == _clearButton) {
        [self hideMenuViews];
        [[BaiduRtcBoardApi sharedManager] clear];
    } else if (sender == _moreButton) {
         [self hidePenMenuViews];
         _moreView.hidden = !_moreView.hidden;
    }
}

- (void)onPageAction:(id)sender {
    if (sender == _prePageButton) {
        [[BaiduRtcBoardApi sharedManager] preBoard];
    } else if (sender == _nextPageButton) {
        [[BaiduRtcBoardApi sharedManager] nextBoard];
    }
}

- (void)changedBoardState:(BoardRoomState *)state {
    _pageIndex = state.pageIndex;
    _totalPageCount = state.totalPageCount;
    _pageLabel.text = [NSString stringWithFormat:@"%d/%d", state.pageIndex + 1, state.totalPageCount];
    _prePageButton.enabled = (state.pageIndex > 0 ? YES : NO) && !_audience;
    _nextPageButton.enabled = ((state.pageIndex + 1) < state.totalPageCount ? YES : NO) && !_audience;
    _undoButton.enabled = state.undoCheck && !_audience;
    _redoButton.enabled = state.redoCheck && !_audience;
    _eraserButton.enabled = state.clearCheck && !_audience;
    _clearButton.enabled = state.clearCheck && !_audience;

    if (state.totalPageCount > 1) {
        _pageView.hidden = NO;
        _boardDelButton.hidden = NO;
    } else {
        _pageView.hidden = YES;
        _boardDelButton.hidden = YES;
    }
    // [self hideSelectButtons];
}

#pragma mark - Private
- (void)stopLoading {
    [_boardView stopLoading];
}

- (int)getRandomNumber:(int)from to:(int)to {
    return (int)(from + (arc4random() % (to - from + 1)));
}

+ (void)configBackgroundView:(UIView *)view hasBorder:(BOOL)hasBorder {
    view.backgroundColor = [UIColor whiteColor];
    if (hasBorder) {
        view.layer.borderWidth = 1;
        view.layer.borderColor = UIColorFromRGBA(0xD3D3D3, 1.0).CGColor;
    }
    view.hidden = hasBorder;
}

@end

@implementation RTCRadioButtonViews {
    UIView *_selView;
    int _viewCount;
    int _defSelected;

    CGRect _frame;
    CGSize _btnRect;
    // CGPoint _btnPoint;
    CGFloat _btnY;
    CGFloat _marginX;
    CGFloat _selPadding;
    BOOL _hasBorder;
}

- (instancetype)initWithFrame:(CGRect)frame
                    hasBorder:(BOOL)hasBorder
                         type:(RTCRadioButtonType)type 
                    viewWidth:(CGFloat)viewWidth
                    viewCount:(int)viewCount
                        index:(int)index
                     delegate:(id<RTCRadioButtonDelegate>)delegate {
    if (self = [super initWithFrame:frame]) {
        _viewCount = viewCount;
        _defSelected = index;

        _hasBorder = hasBorder;
        _frame = frame;
        _btnRect.width = _btnRect.height = viewWidth;
        _btnY = (_frame.size.height - _btnRect.height) / 2;
        _marginX = (_frame.size.width - _viewCount * _btnRect.width) / _viewCount;
        _selPadding = 5.0f;
        self.type = type;
        self.delegate = delegate;

        [RTCBoardCallView configBackgroundView:self hasBorder:hasBorder];
        [self createRadioButtons];
    }
    return self;
}

- (void)createRadioButtons {
    // selected rectangle
    _selView = [[UIView alloc] initWithFrame:CGRectZero];
    _selView.opaque = NO;
    _selView.backgroundColor = [UIColor clearColor];
    _selView.layer.borderWidth = 1;
    _selView.layer.cornerRadius = 4.0;
    _selView.layer.borderColor = UIColorFromRGBA(0x4876FF, 1.0).CGColor;
    _selView.frame = [self makeSelectRect:_defSelected];
    [self addSubview:_selView];

    // radio buttons
    for (int i = 0; i < _viewCount; i++) {
        UIButton *btn = [UIButton buttonWithType:UIButtonTypeCustom];

        btn.tag = i;
        btn.layer.cornerRadius = 4.0;
        btn.layer.masksToBounds = YES;
        CGFloat btnX = _marginX + i * (_btnRect.width + _marginX) - _marginX / 2;
        btn.frame = CGRectMake(btnX, _btnY, _btnRect.width, _btnRect.height);
        [self addSubview:btn];

        [btn addTarget:self action:@selector(onSelectRadio:) forControlEvents:UIControlEventTouchUpInside];
        if (self.delegate && [self.delegate respondsToSelector:@selector(onRadioButtonAdd:button:index:)]) {
            [self.delegate onRadioButtonAdd:self.type button:btn index:i];
        }
    }
}

- (CGRect)makeSelectRect:(int)index {
    CGFloat selX = _marginX / 2 - _selPadding + (_btnRect.width + _marginX) * index;
    CGFloat selY = _btnY - _selPadding;
    CGFloat selW = _btnRect.width + _selPadding * 2;
    CGFloat selH = _btnRect.height + _selPadding * 2;
    return CGRectMake(selX, selY, selW, selH);
}

- (void)onSelectRadio:(UIButton *)sender {
    _selView.frame = [self makeSelectRect:(int)sender.tag];
    if (self.delegate && [self.delegate respondsToSelector:@selector(onRadioButtonSelect:button:index:)]) {
        [self.delegate onRadioButtonSelect:self.type button:sender index:(int)sender.tag];
    }
}

@end

@interface RTCBoardSettingView () <RTCRadioButtonDelegate>

@property (nonatomic, strong) UILabel       *bgAlphaLabel;
@property (nonatomic, strong) UISlider      *bgAlphaSlider;
@property (nonatomic, strong) UILabel       *bgAlphaValLabel;

@property (nonatomic, strong) UILabel       *bgColorLabel;

// joining room with parameters of audience role.
// in the room during collaboration switching between audience and communicator is complicated.
@property (nonatomic, strong) UIImageView   *roleView;

@property (nonatomic, strong) UILabel       *statsLabel;
@property (nonatomic, strong) UISwitch      *statsSwitch;

@property (nonatomic, strong) UILabel       *gridLabel;
@property (nonatomic, strong) UISwitch      *gridSwitch;

@property (nonatomic, strong) UIButton      *snapshotButton;
@end

@implementation RTCBoardSettingView {
    UIView *_bgView;
    CGRect _bgframe;

    UIView *_bgColorView;
    int _defTransparency;
    int _defBgColorIndex;
}

- (UILabel *)createLabelForTitle:(NSString *)title {
    UILabel *label = [[UILabel alloc] initWithFrame:CGRectZero];
    label.adjustsFontSizeToFitWidth = YES;
    label.textColor = [UIColor darkGrayColor];
    label.text = title;
    return label;
}

- (instancetype)initWithFrame:(CGRect)frame
                 transparency:(int)transparency
                   colorIndex:(int)colorIndex
                     delegate:(id<RTCBoardSettingDelegate>)delegate {
    if (self = [super initWithFrame:frame]) {
        _bgframe = frame;
        _defTransparency = transparency;
        _defBgColorIndex = colorIndex;
        self.delegate = delegate;

        [RTCBoardCallView configBackgroundView:self hasBorder:YES];
        [self createView];
    }
    return self;
}

- (void)createView {
    self.bgAlphaLabel    = [self createLabelForTitle:@"Transparency:"];
    self.bgAlphaValLabel = [self createLabelForTitle:@(_defTransparency).stringValue];
    self.bgAlphaSlider   = [self createSliderMinVal:0 maxVal:100 defVal:_defTransparency];
    self.statsLabel      = [self createLabelForTitle:@"Stats:"];
    self.statsSwitch     = [self createSwitchWithTag:1];
    self.gridLabel       = [self createLabelForTitle:@"Grid:"];
    self.gridSwitch      = [self createSwitchWithTag:2];
    self.snapshotButton  = [self createXButtonForTitle:@"Snapshot" titleColor:[UIColor whiteColor] bgColor:[UIColor systemGreenColor]];
    NSString *roleName   = [[RTCSettingsModel sharedInstance] currentBoardAudienceFromStore] ? @"role1.png" : @"role0.png";
    self.roleView        = [[UIImageView alloc] initWithImage:[UIImage imageNamed:roleName]];

    [self addSubview:self.bgAlphaLabel];
    [self addSubview:self.bgAlphaValLabel];
    [self addSubview:self.bgAlphaSlider];
    [self addSubview:self.statsLabel];
    [self addSubview:self.statsSwitch];
    [self addSubview:self.gridLabel];
    [self addSubview:self.gridSwitch];
    [self addSubview:self.snapshotButton];
    [self addSubview:self.roleView];

    [self layoutViews];
}

- (void)layoutViews {
    // slide bar
    CGFloat topY = 30, margin = 12, height = 24, alphaLabelW = 100, alphaValLabelW = 30;
    CGFloat sliderWidth = SCREEN_WIDTH - alphaLabelW - alphaValLabelW - margin * 4;
    self.bgAlphaLabel.frame = CGRectMake(margin, topY, alphaLabelW, height);
    CGFloat sliderX = CGRectGetMaxX(self.bgAlphaLabel.frame) + margin;
    self.bgAlphaSlider.frame = CGRectMake(sliderX, topY, sliderWidth, height);
    self.bgAlphaValLabel.frame = CGRectMake(CGRectGetMaxX(self.bgAlphaSlider.frame) + margin, topY, alphaValLabelW, height);

    // select background
    CGFloat marginX = 20, colorLabelW = 86, colorFrameH = 70;
    CGFloat colorFrameW = SCREEN_WIDTH - colorLabelW - marginX * 2;
    topY += 30;
    self.bgColorLabel    = [self createLabelForTitle:@"Background:"];
    self.bgColorLabel.frame = CGRectMake(sliderX - colorLabelW, topY + colorFrameH / 2 - height / 2, colorLabelW, height);
    [self addSubview:self.bgColorLabel];
    CGRect colorFrame = CGRectMake(sliderX, topY, colorFrameW, colorFrameH);
    UIView *colorView = [[RTCRadioButtonViews alloc] initWithFrame:colorFrame hasBorder:NO 
                                                              type:RTC_RADIO_BUTTON_TYPE_COLOR
                                                         viewWidth:24 viewCount:6 index:_defBgColorIndex delegate:self];
    [self addSubview:colorView];

    // stats and snapshot
    topY += colorFrameH;
    CGFloat buttonH = 36;

    self.statsLabel.frame = CGRectMake(sliderX - 40, topY + (buttonH - height + 6) / 2, 40, height);
    self.statsSwitch.frame = CGRectMake(CGRectGetMaxX(self.statsLabel.frame) + margin, topY + (buttonH - height) / 2, 50, height);
    self.gridLabel.frame = CGRectMake(CGRectGetMaxX(self.statsSwitch.frame) + margin, topY + (buttonH - height + 6) / 2, 34, height);
    self.gridSwitch.frame = CGRectMake(CGRectGetMaxX(self.gridLabel.frame) + margin, topY + (buttonH - height) / 2, 50, height);

    CGFloat snapshotW = 120;
    topY += 60;
    CGFloat roleL = 60;
    self.snapshotButton.frame = CGRectMake(SCREEN_WIDTH / 2 - snapshotW / 2 + marginX, topY, snapshotW, buttonH);
    self.roleView.frame = CGRectMake(CGRectGetMinX(self.snapshotButton.frame) - roleL - marginX, topY - (roleL - buttonH) / 2, roleL, roleL);
}

- (void)onRadioButtonAdd:(RTCRadioButtonType)type button:(UIButton *)button index:(int)index {
    if (type == RTC_RADIO_BUTTON_TYPE_COLOR) {
        button.backgroundColor = kBackgroundColorTable()[index];
    }
}

- (void)onRadioButtonSelect:(RTCRadioButtonType)type button:(UIButton *)button index:(int)index {
    if (self.delegate && [self.delegate respondsToSelector:@selector(onBackgroundColorChanged:)]) {
        [self.delegate onBackgroundColorChanged:index];
    }
}

- (UISlider *)createSliderMinVal:(CGFloat)minVal maxVal:(CGFloat)maxVal defVal:(CGFloat)defVal {
    UISlider *slider = [[UISlider alloc] init];
    slider.minimumValue = minVal;
    slider.maximumValue = maxVal;
    slider.value = defVal;
    slider.continuous = YES;
    [slider addTarget:self action:@selector(sliderValueChanged:) forControlEvents:UIControlEventValueChanged];
    return slider;
}

- (UISwitch *)createSwitchWithTag:(NSInteger)tag {
    UISwitch *switchView = [[UISwitch alloc] initWithFrame:CGRectZero];
    switchView.tag = tag;
    switchView.onTintColor = [UIColor systemBlueColor];
    [switchView addTarget:self action:@selector(switchChanged:) forControlEvents:UIControlEventValueChanged];
    return switchView;
}

- (UIButton *)createXButtonForTitle:(NSString *)title titleColor:(UIColor *)titleColor bgColor:(UIColor *)bgColor {
    UIButton *button = [UIButton buttonWithType:UIButtonTypeCustom];
    button.layer.masksToBounds = YES;
    button.backgroundColor = bgColor;
    [button setTitle:title forState:UIControlStateNormal];
    [button setTitleColor:titleColor forState:UIControlStateNormal];
    button.titleLabel.font = [UIFont fontWithName:@"PingFangSC-Regular" size:14];
    [button addTarget:self action:@selector(onButtonClicked:) forControlEvents:UIControlEventTouchUpInside];
    return button;
}

- (void)onButtonClicked:(UIButton *)sender {
    if (self.delegate && [self.delegate respondsToSelector:@selector(onSnapshotTaken)]) {
        [self.delegate onSnapshotTaken];
    }
}

- (void)switchChanged:(UISwitch *)sender {
    switch (sender.tag) {
        case 0: // audience
            break;
        case 1: // stats
            [[BaiduRtcBoardApi sharedManager] stats];
            break;
        case 2: // grid
            [[BaiduRtcBoardApi sharedManager] grid];
            break;
        default:
            break;
    }
}

- (void)sliderValueChanged:(UISlider *)sender {
    self.bgAlphaValLabel.text = [NSString stringWithFormat:@"%@", @((int)sender.value)];
    if (self.delegate && [self.delegate respondsToSelector:@selector(onBackgroundTransparencyChanged:)]) {
        [self.delegate onBackgroundTransparencyChanged:(int)sender.value];
    }
}

@end
