//
//  RTCSettingsAudioController.m
//  rtc_sdk_ios
//
//  Created by Sun,Jian(ACU-T102) on 2018/12/31.
//  Copyright © 2018年 Sun,Jian(ACU-T102). All rights reserved.
//

#import "RTCSettingsAudioController.h"
#import "RTCSettingsTextTableViewCell.h"
#import "BaiduRtcRoom/BaiduRtcRoom.h"
#import "RTCSettingsModel.h"

NS_ASSUME_NONNULL_BEGIN

static NSArray<NSString *> *audioProfileSaticValues(){
    return @[@"DEFAULT", @"LOW_QUALITY", @"STANDARD", @"HIGH_QUALITY", @"STEREO_HIGH_QUALITY", @"SUPER_QUALITY", @"STEREO_SUPER_QUALITY"];
}

static NSArray<NSString *> *audioScenarioSaticValues(){
    return @[@"BRTC_AUDIO_SCENARIO_DEFAULT", @"BRTC_AUDIO_SCENARIO_SPEECH", @"BRTC_AUDIO_SCENARIO_MUSIC", @"BRTC_AUDIO_SCENARIO_METAVERSE"];
}

typedef NS_ENUM(int, RTCSettingsSections) {
    RTCSettingsSectionAudioCode = 0,
    RTCSettingsSectionAudioSettings,
    RTCSettingsSectionsAudioDisposeSet,
    __RTCSettingsSectionsEnd
};

typedef NS_ENUM(int, RTCAudioSectionOptions) {
    RTCAudioSectionCodec = 0,
    RTCAudioSectionProfile,
    RTCAudioSectionScenario,
    RTCAudioSectionBgPushMode,
};

typedef NS_ENUM(int, RTCAudioDisposeSettingOption) {
    RTCAudioSettingEchoCancellation = 0,
    RTCAudioSettingNoiseSuppression,
    RTCAudioSettingVoiceGain,
};

typedef NS_ENUM(int, RTCAudioSettingsOptions) {
    RTCAudioSettingsAudioOnly = 0,
    RTCAudioSettingsUseManualAudioConfig,
    RTCAudioSettingsCreateAecDump,
    RTCAudioSettingsUseLevelController,
};

@interface RTCSettingsAudioController ()  {
}
@property (nonatomic, strong) RTCSettingsModel *settingsModel;
@end

@implementation RTCSettingsAudioController

- (instancetype)initWithStyle:(UITableViewStyle)style
                settingsModel:(RTCSettingsModel *)settingsModel {
    self = [super initWithStyle:style];
    if (self) {
        self.settingsModel = settingsModel;
    }
    return self;
}

- (void)dealloc {
    [_settingsModel sync];
}

- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView {
    return __RTCSettingsSectionsEnd;
}

- (NSString *)getAudioProfileStr:(BRTCAudioProfileType)type {
    return audioProfileSaticValues()[type];
}

- (NSString *)getAudioScenarioStr:(BRTCAudioScenario)scenario {
    return audioScenarioSaticValues()[scenario];
}

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section {
    switch (section) {
        case RTCSettingsSectionAudioSettings:
            return 2; //音频去掉无关的选项  AecDump文件使用分级控制器，剩余：仅仅开启音频和手动音频配置
        case RTCSettingsSectionAudioCode:
            return 4;
        case RTCSettingsSectionsAudioDisposeSet:
            return 3;
        default:
            return 1;
    }
}

- (nullable NSString *)tableView:(UITableView *)tableView
         titleForHeaderInSection:(NSInteger)section {
    switch (section) {
         case RTCSettingsSectionAudioSettings:
            return @"音频";
        case RTCSettingsSectionAudioCode:
            return @"音频";
        case RTCSettingsSectionsAudioDisposeSet:
            return @"音频处理";
        default:
            return @"";
    }
}

- (UITableViewCell *)tableView:(UITableView *)tableView
         cellForRowAtIndexPath:(NSIndexPath *)indexPath {
    switch (indexPath.section) {
        case RTCSettingsSectionAudioSettings:
            return [self audioSettingsTableViewCellForTableView:tableView atIndexPath:indexPath];
        case RTCSettingsSectionAudioCode:
            return [self AudioCodecTableViewCellForTableView:tableView atIndexPath:indexPath];
        case RTCSettingsSectionsAudioDisposeSet:
            return  [self audioDiposeTableViewCellForTableView:tableView atIndexPath:indexPath];
        default:
            return [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
                                          reuseIdentifier:@"identifier"];
    }
}

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath {
    switch (indexPath.section) {
        case RTCSettingsSectionAudioCode:
            [self tableView:tableView didSelectAudioCodecCellAtIndexPath:indexPath];
            [self.tableView reloadData];
            break;
    }
}

- (UITableViewCell *)AudioCodecTableViewCellForTableView:(UITableView *)tableView
                                             atIndexPath:(NSIndexPath *)indexPath {
    NSString *title = nil, *key = nil;
    switch (indexPath.row) {
        case RTCAudioSectionCodec:
            title = @"编解码";
            key = [_settingsModel currentAudioCodecSettingFromStore];
            break;
        case RTCAudioSectionProfile: {
            title = @"Profile";
            int index = (int) [_settingsModel currentAudioProfileSettingFromStore].integerValue;
            key = [self getAudioProfileStr:(BRTCAudioProfileType)index];
            break;
        }
        case RTCAudioSectionScenario: {
            title = @"Scenario";
            int index = (int) [_settingsModel currentAudioScenarioSettingFromStore].integerValue;
            key = [self getAudioScenarioStr:index];
            break;
        }
        case RTCAudioSectionBgPushMode: {
            title = @"后台推流";
            int index = (int) [[_settingsModel currentCaptureVideoBgPushModeSettingFromStore] integerValue];
            key = [_settingsModel availableVideoBgPushMode][index];
            break;
        }
    }
    return [self tableCellForTableView:tableView
                            identifier:@"RTCSettingsAudioCodecCellIdentifier"
                                 title:title
                                   key:key
                           atIndexPath:indexPath];
}

- (void)tableView:(UITableView *)tableView
        didSelectAudioCodecCellAtIndexPath:(NSIndexPath *)indexPath {
    NSArray<NSString *> *arraySelect = nil;
    switch (indexPath.row) {
        case RTCAudioSectionCodec:
            arraySelect = [_settingsModel availableAudioCodecs];
            break;
        case RTCAudioSectionProfile:
            arraySelect = audioProfileSaticValues();
            break;
        case RTCAudioSectionScenario:
            arraySelect = audioScenarioSaticValues();
            break;
        case RTCAudioSectionBgPushMode:
            arraySelect = [_settingsModel availableVideoBgPushMode];
            break;
    }
    [self tableView:tableView arraySelect:arraySelect success:^(int index) {
        switch (indexPath.row) {
            case RTCAudioSectionCodec:
                [self.settingsModel storeAudioCodecSetting:[self.settingsModel availableAudioCodecs][index]];
                break;
            case RTCAudioSectionProfile:
                [self.settingsModel storeAudioProfileSetting:@(index)];
                break;
            case RTCAudioSectionScenario:
                [self.settingsModel storeAudioScenarioSetting:@(index)];
                break;
            case RTCAudioSectionBgPushMode:
                [self.settingsModel storeCaptureVideoBgPushModeSetting:@(index)];
                break;
        }
    }];
}

#pragma mark - Table view delegate(audio dispose)
- (UITableViewCell *)audioDiposeTableViewCellForTableView:(UITableView *)tableView
                                        atIndexPath:(NSIndexPath *)indexPath {
    NSString *dequeueIdentifier = @"RTCSettingsAudioDisposeViewCellIdentifier";
    UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:dequeueIdentifier];
    if (!cell) {
        cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
                                    reuseIdentifier:dequeueIdentifier];
        cell.selectionStyle = UITableViewCellSelectionStyleNone;
        UISwitch *switchView = [[UISwitch alloc] initWithFrame:CGRectZero];
        switchView.tag = indexPath.row;
        [switchView addTarget:self
                    action:@selector(audioDisposeSwitchChanged:)
            forControlEvents:UIControlEventValueChanged];
        cell.accessoryView = switchView;
    }
    cell.textLabel.text = [self labelForAudioDisposeSettingAtIndexPathRow:indexPath.row];
    UISwitch *switchView = (UISwitch *)cell.accessoryView;
    switchView.on = [self valueForAudioDisposeSettingAtIndexPathRow:indexPath.row];
    return cell;
}

- (void)audioDisposeSwitchChanged:(UISwitch *) sender {
    switch (sender.tag) {
        case RTCAudioSettingEchoCancellation: {
            [_settingsModel storeEchoCancellationSetting:sender.isOn];
            break;
        }
        case RTCAudioSettingNoiseSuppression: {
            [_settingsModel storeNoiseSuppressionSetting:sender.isOn];
            break;
        }
        case RTCAudioSettingVoiceGain: {
            [_settingsModel storeVoiceGainSetting:sender.isOn];
            break;
        }
        default:
            break;
    }
}

#pragma mark - Table view delegate(Audio settings)
- (UITableViewCell *)audioSettingsTableViewCellForTableView:(UITableView *)tableView
                                                atIndexPath:(NSIndexPath *)indexPath {
    NSString *dequeueIdentifier = @"RTCSettingsAudioSettingsCellIdentifier";
    UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:dequeueIdentifier];
    if (!cell) {
        cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
                                      reuseIdentifier:dequeueIdentifier];
        cell.selectionStyle = UITableViewCellSelectionStyleNone;
        UISwitch *switchView = [[UISwitch alloc] initWithFrame:CGRectZero];
        switchView.tag = indexPath.row;
        [switchView addTarget:self
                       action:@selector(audioSettingSwitchChanged:)
             forControlEvents:UIControlEventValueChanged];
        cell.accessoryView = switchView;
    }

    cell.textLabel.text = [self labelForAudioSettingAtIndexPathRow:indexPath.row];
    UISwitch *switchView = (UISwitch *)cell.accessoryView;
    switchView.on = [self valueForAudioSettingAtIndexPathRow:indexPath.row];
    return cell;
}

- (NSString *)labelForAudioSettingAtIndexPathRow:(NSInteger)setting {
    switch (setting) {
        case RTCAudioSettingsAudioOnly:
            return @"仅仅开启音频";
        case RTCAudioSettingsCreateAecDump:
            return @"创建AecDump文件";
        case RTCAudioSettingsUseLevelController:
            return @"使用分级控制器";
        case RTCAudioSettingsUseManualAudioConfig:
            return @"使用手动音频配置";
        default:
            return @"";
    }
}
- (NSString *)labelForAudioDisposeSettingAtIndexPathRow:(NSInteger)setting {
    switch (setting) {
        case RTCAudioSettingEchoCancellation:
            return @"回声消除";
        case RTCAudioSettingNoiseSuppression:
            return @"噪音抑制";
        case RTCAudioSettingVoiceGain:
            return @"人声增益";
        default:
            return @"";
    }
}

- (BOOL)valueForAudioDisposeSettingAtIndexPathRow:(NSInteger)setting {
    switch (setting) {
        case RTCAudioSettingEchoCancellation:
            return ![_settingsModel currentEchoCancellationSettingFromStore];
        case RTCAudioSettingNoiseSuppression:
            return [_settingsModel currentNoiseSuppressionSettingFromStore];
        case RTCAudioSettingVoiceGain:
            return [_settingsModel currentVoiceGainSettingFromStore];
        default:
            return NO;
    }
}

- (BOOL)valueForAudioSettingAtIndexPathRow:(NSInteger)setting {
    switch (setting) {
        case RTCAudioSettingsAudioOnly:
            return ![_settingsModel currentHasVideoSettingFromStore];
        case RTCAudioSettingsCreateAecDump:
            return [_settingsModel currentCreateAecDumpSettingFromStore];
        case RTCAudioSettingsUseLevelController:
            return [_settingsModel currentUseLevelControllerSettingFromStore];
        case RTCAudioSettingsUseManualAudioConfig:
            return [_settingsModel currentUseManualAudioConfigSettingFromStore];
        default:
            return NO;
    }
}

- (void)audioSettingSwitchChanged:(UISwitch *)sender {
    switch (sender.tag) {
        case RTCAudioSettingsAudioOnly: {
            [_settingsModel storeHasVideoSetting:!sender.isOn];
            break;
        }
        case RTCAudioSettingsCreateAecDump: {
            [_settingsModel storeCreateAecDumpSetting:sender.isOn];
            break;
        }
        case RTCAudioSettingsUseLevelController: {
            [_settingsModel storeUseLevelControllerSetting:sender.isOn];
            break;
        }
        case RTCAudioSettingsUseManualAudioConfig: {
            [_settingsModel storeUseManualAudioConfigSetting:sender.isOn];
            break;
        }
        default:
            break;
    }
}

#pragma mark - View lifecycle
- (void)viewDidLoad {
    [super viewDidLoad];
    self.title = @"音频设置";
    [self addDoneBarButton];
}

@end
NS_ASSUME_NONNULL_END

