//
//  RTCCallingSettingScreenView.m
//  rtc_sdk_ios
//
//  Created by 曹士谦 on 2022/8/23.
//  Copyright © 2022 Sun,Jian(ACU-T102). All rights reserved.
//

#import "RTCCallingSettingScreenView.h"
#import "UIView+BBAUtility.h"
#import <BaiduRtcRoom/BaiduRtcRoomApiDefines.h>

static int kDefaultVolume = 100;
static int kDefaultFPS = 15;
static NSArray<NSString *> *kResolutionSettings() {
    return  @[ @"1080x1920", @"720x1280", @"480x640" ];
}
static NSArray<NSNumber *> *kBitrateSettings() {
    return  @[ @(2500), @(1500), @(500) ];
}

@interface RTCCallingSettingScreenView ()<UITextFieldDelegate>

@property (nonatomic, strong) UILabel       *fpsLabel;
@property (nonatomic, strong) UITextField   *fpsField;
@property (nonatomic, strong) UILabel       *resLabel;
@property (nonatomic, strong) UITextField   *resField;
@property (nonatomic, strong) UILabel       *bitrateLabel;
@property (nonatomic, strong) UITextField   *bitrateField;

@property (nonatomic, strong) UILabel       *volumeLabel;
@property (nonatomic, strong) UISlider      *volumeSlider;
@property (nonatomic, strong) UILabel       *volumeValLabel;

@property (nonatomic, strong) UILabel       *captureAudioLabel;
@property (nonatomic, strong) UISwitch      *captureAudioSwitch;
@property (nonatomic, strong) UILabel       *captureVideoLabel;
@property (nonatomic, strong) UISwitch      *captureVideoSwitch;
@property (nonatomic, strong) UIButton      *applyButton;
@property (nonatomic, strong) UIButton      *snapshotButton;

@property (nonatomic, strong) BRTCScreenShareParams *params;

@end

@implementation RTCCallingSettingScreenView

- (instancetype)initWithFrame:(CGRect)frame {
    if (self = [super initWithFrame:frame]) {
        [self initView];
        _params = [BRTCScreenShareParams new];
        [self applyParams];
    }
    return self;
}

- (void)initView {
    self.layer.masksToBounds = YES;
    self.backgroundColor = [UIColor whiteColor];
    
    // 视频参数
    self.fpsLabel       = [self createLabelForTitle:@"帧率"];
    self.resLabel       = [self createLabelForTitle:@"分辨率"];
    self.bitrateLabel   = [self createLabelForTitle:@"码率(KBPS)"];
    [self addSubview:self.fpsLabel];
    [self addSubview:self.resLabel];
    [self addSubview:self.bitrateLabel];
    
    NSString *defaultFps = @(kDefaultFPS).stringValue;
    NSString *defaultRes = kResolutionSettings()[0];
    NSString *defaultBit = kBitrateSettings()[0].stringValue;
    self.fpsField       = [self createTextField:defaultFps text:defaultFps];
    self.resField       = [self createTextField:defaultRes text:defaultRes];
    self.bitrateField   = [self createTextField:defaultBit text:defaultBit];
    [self addSubview:self.fpsField];
    [self addSubview:self.resField];
    [self addSubview:self.bitrateField];
    
    // 音频参数
    self.volumeLabel    = [self createLabelForTitle:@"音量"];
    self.volumeValLabel = [self createLabelForTitle:@(kDefaultVolume).stringValue];
    self.volumeSlider   = [self createSliderMinVal:0 maxVal:100 defVal:kDefaultVolume];
    [self addSubview:self.volumeLabel];
    [self addSubview:self.volumeValLabel];
    [self addSubview:self.volumeSlider];
    
    // 是否采集流
    self.captureAudioLabel  = [self createLabelForTitle:@"音频"];
    self.captureVideoLabel  = [self createLabelForTitle:@"视频"];
    self.captureAudioSwitch = [self createSwitchWithTag:0];
    self.captureVideoSwitch = [self createSwitchWithTag:1];
    [self addSubview:self.captureAudioLabel];
    [self addSubview:self.captureAudioSwitch];
    [self addSubview:self.captureVideoLabel];
    [self addSubview:self.captureVideoSwitch];
    
    self.applyButton = [self createButtonForTitle:@"应用" titleColor:[UIColor whiteColor] bgColor:[UIColor systemBlueColor]];
    [self addSubview:self.applyButton];
    
    // 截图
    self.snapshotButton = [self createButtonForTitle:@"截图" titleColor:[UIColor whiteColor] bgColor:[UIColor systemGreenColor]];
    [self addSubview:self.snapshotButton];
}

- (void)layoutSubviews {
    [super layoutSubviews];
    
    CGFloat width = self.frame.size.width;
    
    CGFloat paddingHor = 10.f, margin = 5.f;
    CGFloat height = 30.f;
    CGFloat top = 5.f, bottom = 0.f;
    bottom = CGRectGetMaxY(self.captureAudioLabel.frame);
    
    CGFloat lblWidth = 100.f, fieldWidth = width - paddingHor * 2 - lblWidth - margin;
    self.fpsLabel.frame     = CGRectMake(paddingHor, top, lblWidth, height);
    self.resLabel.frame     = CGRectMake(paddingHor, top + height + margin, lblWidth, height);
    self.bitrateLabel.frame = CGRectMake(paddingHor, top + (height + margin) * 2, lblWidth, height);
    self.fpsField.frame     = CGRectMake(lblWidth + margin, top, fieldWidth, height);
    self.resField.frame     = CGRectMake(lblWidth + margin, top + height + margin, fieldWidth, height);
    self.bitrateField.frame = CGRectMake(lblWidth + margin, top + (height + margin) * 2, fieldWidth, height);
    bottom = CGRectGetMaxY(self.bitrateLabel.frame);
    
    top = bottom + margin;
    CGFloat sliderWidth = width - paddingHor * 2 - 50 - 30 - margin * 2;
    self.volumeLabel.frame = CGRectMake(paddingHor, top, 50, height);
    self.volumeSlider.frame = CGRectMake(CGRectGetMaxX(self.volumeLabel.frame) + margin, top, sliderWidth, height);
    self.volumeValLabel.frame = CGRectMake(CGRectGetMaxX(self.volumeSlider.frame) + margin, top, 30, height);
    bottom = CGRectGetMaxY(self.volumeLabel.frame);
    
    top = bottom + margin;
    self.captureAudioLabel.frame    = CGRectMake(paddingHor, top, 30, height);
    self.captureAudioSwitch.frame   = CGRectMake(CGRectGetMaxX(self.captureAudioLabel.frame) + margin, top, 0, height);
    self.captureVideoLabel.frame    = CGRectMake(CGRectGetMaxX(self.captureAudioSwitch.frame) + margin, top, 30, height);
    self.captureVideoSwitch.frame   = CGRectMake(CGRectGetMaxX(self.captureVideoLabel.frame) + margin, top, 0, height);
    self.applyButton.frame          = CGRectMake(CGRectGetMaxX(self.captureVideoSwitch.frame) + margin, top + 2, 50, height);
    bottom = CGRectGetMaxY(self.captureAudioLabel.frame);
    
    top = bottom + margin;
    self.snapshotButton.frame       = CGRectMake(paddingHor, top, 100, height);
}

- (void)applyParams {
    self.captureAudioSwitch.on = self.params.enableAudioCapture;
    self.captureVideoSwitch.on = self.params.enableVideoCapture;
    
}

#pragma mark - Event

- (BOOL)textFieldShouldBeginEditing:(UITextField *)textField {
    if (textField == self.resField) {
        // 简易编辑分辨率，在几个选择中切换
        [self switchResolution];
        return NO;
    }
    return YES;
}

- (void)textFieldDidEndEditing:(id)sender {
    [self endEditing:YES];
}

- (void)switchChanged:(UISwitch *)sender {
    switch (sender.tag) {
        case 0: // capture audio
            self.params.enableAudioCapture = sender.on;
            break;
        case 1: // capture video
            self.params.enableVideoCapture = sender.on;
            break;
        default:
            break;
    }
}

- (void)switchResolution {
    NSInteger index = 0;
    NSArray<NSString *> *resList = kResolutionSettings();
    
    if (self.resField.text.length) {
        NSInteger currentIndex = [resList indexOfObject:self.resField.text];
        if (currentIndex == resList.count - 1) {
            self.resField.text = nil;
            self.bitrateField.text = nil;
            return;
        }
        if (currentIndex >= 0) {
            index = (currentIndex + 1) % resList.count;
        }
    }
    
    self.resField.text = [resList objectAtIndex:index];
    self.bitrateField.text = [kBitrateSettings() objectAtIndex:index].stringValue;
}

- (void)sliderValueChanged:(UISlider *)sender {
    self.params.audioCaptureParams.volume = (int)sender.value;
    self.volumeValLabel.text = [NSString stringWithFormat:@"%@", @(self.params.audioCaptureParams.volume)];
}

- (void)buttonClicked:(UIButton *)sender {
    
    if (sender == self.applyButton) {
        if (self.fpsField.text.intValue > 0) {
            self.params.videoCaptureParams.videoFps = self.fpsField.text.intValue;
        }
        
        if (self.resField.text.length > 0) {
            NSString *resStr = self.resField.text;
            self.params.videoCaptureParams.videoWidth = [self resolutionComponentAtIndex:0 inString:resStr];
            self.params.videoCaptureParams.videoHeight = [self resolutionComponentAtIndex:1 inString:resStr];
        }
        
        if (self.bitrateField.text.intValue > 0) {
            self.params.videoCaptureParams.videoBitrate = self.bitrateField.text.intValue;
        }
        
        if ([self.delegate respondsToSelector:@selector(screenSettingParamsUpdated:)]) {
            [self.delegate screenSettingParamsUpdated:self.params];
        }
        return;
    }
    
    if (sender == self.snapshotButton) {
        if ([self.delegate respondsToSelector:@selector(screenSettingSnapshot)]) {
            [self.delegate screenSettingSnapshot];
        }
    }
}

- (int)resolutionComponentAtIndex:(int)index inString:(NSString *)resolution {
    if (index != 0 && index != 1) {
        return 0;
    }
    NSArray<NSString *> *components = [resolution componentsSeparatedByString:@"x"];
    if (components.count != 2) {
        return 0;
    }
    return components[index].intValue;
}

#pragma mark - View

- (UIButton *)createButtonForTitle:(NSString *)title titleColor:(UIColor *)titleColor bgColor:(UIColor *)bgColor {
    UIButton *btn = [UIButton brtc_buttonWithTitle:title titleColor:titleColor bgColor:bgColor fontSize:14.f cornerRadius:3.f];
    [btn addTarget:self action:@selector(buttonClicked:) forControlEvents:UIControlEventTouchUpInside];
    return btn;
}

- (UITextField *)createTextField:(NSString *)pholder text:(NSString *)context {
    UITextField *textField = [UITextField brtc_textFieldWithPlaceholder:pholder hasBorder:YES];
    textField.delegate = self;
    [textField addDoneButtonWithFrame:CGRectMake(0, 0, self.bounds.size.width, 50) target:self action:@selector(textFieldDidEndEditing:)];
    return textField;
}

- (UILabel *)createLabelForTitle:(NSString *)title {
    return [UILabel brtc_labelWithTitle:title color:[UIColor blackColor] fontSize:14.f];
}

- (UISwitch *)createSwitchWithTag:(NSInteger)tag {
    UISwitch *switchView = [[UISwitch alloc] initWithFrame:CGRectZero];
    switchView.tag = tag;
    switchView.onTintColor = [UIColor systemBlueColor];
    [switchView addTarget:self action:@selector(switchChanged:) forControlEvents:UIControlEventValueChanged];
    return switchView;
}

- (UISlider *)createSliderMinVal:(CGFloat)minVal maxVal:(CGFloat)maxVal defVal:(CGFloat)defVal {
    UISlider *slider = [[UISlider alloc] init];
    slider.minimumValue = minVal;
    slider.maximumValue = maxVal;
    slider.value = defVal;
    slider.continuous = NO;
    [slider addTarget:self action:@selector(sliderValueChanged:) forControlEvents:UIControlEventValueChanged];
    return slider;
}

@end
