//
//  BDCloudAVRTCRoomAudioDevice.m
//  svlapp
//
//  Created by Fu,Sheng(ACG VCP) on 2021/7/9.
//  Copyright © 2021 fusheng. All rights reserved.
//

#import "BDCloudAVRTCRoomAudioDeviceFactory.h"

@interface BDCloudAVRTCRoomAudioRecordDevice : NSObject<BaiduRtcRoomApiAudioExternalRecordDevice, BDCloudAVPCMConverterDelegate>

@property (nonatomic, weak) id<BaiduRtcRoomApiAudioExternalRecordClient> client;
@property (nonatomic, strong) BDCloudAVPCMConverter *converter;
@property (nonatomic, strong) BDCloudAVAudioPTSHelper *ptsHelper;
@property (nonatomic, strong) BDCloudAVAudioBufferList *bufferList;
@property (nonatomic, strong) AVAudioFormat *outputFormat;
@property (atomic, assign) BOOL started;

- (instancetype)initWithClient:(id<BaiduRtcRoomApiAudioExternalRecordClient>)client;
- (void)onIncommingBuffer:(CMSampleBufferRef)frame;
@end

@implementation BDCloudAVRTCRoomAudioRecordDevice
- (instancetype)initWithClient:(id<BaiduRtcRoomApiAudioExternalRecordClient>)client {
    if (self = [super init]) {
        _client = client;
    }
    return self;
}

- (void)onIncommingBuffer:(CMSampleBufferRef)frame {
    do {
        if (!self.started) {
            break;
        }
        BDCloudAVCMSampleBuffer* buffer = [BDCloudAVCMSampleBuffer fromCMSampleBuffer:frame];
        BDCloudAVAudioBufferList* abl = buffer.abl;
        AVAudioFormat *inputFormat = [[AVAudioFormat alloc] initWithStreamDescription:abl.asbd];
        if ([inputFormat isEqual:self.outputFormat]) {
            [self pcmDataReady:abl];
        } else {
            if (!_converter) {
                BDCloudAVPCMConverter *converter = [[BDCloudAVPCMConverter alloc] init];
                converter.outputFormat = self.outputFormat;
                converter.frames = 1024;
                converter.delgate = self;
                _converter = converter;
            }
            if ([_converter setup:abl.asbd] != noErr) {
                break;
            }
            
            if (!_ptsHelper) {
                _ptsHelper = [[BDCloudAVAudioPTSHelper alloc] init];
            }
            CMTime current = buffer.pts;
            if (![_ptsHelper validate:current sampleRate:_converter.inputFormat.sampleRate]) {
                [_converter clear];
            }
            _ptsHelper.lastPTS = current;
            _ptsHelper.lastSamples = abl.samples;
            _ptsHelper.remainder = _converter.fill;
            [_converter convert:abl];
        }
    } while (NO);
}

#pragma mark --BDCloudAVPCMConverterDelegate
static AudioUnitRenderActionFlags flag = kAudioOfflineUnitRenderAction_Render;
static AudioTimeStamp timeStamp = {0};
- (void)pcmDataReady:(BDCloudAVAudioBufferList *)abl {
    if (self.started) {
        self.bufferList = abl;
        [self.client onRecordData:&flag
                             time:&timeStamp
                           busNum:0
                        numFrames:abl.samples
                           ioData:abl.abl];
    }
}

#pragma mark --BaiduRtcRoomApiAudioExternalRecordDevice
- (BOOL)initializeExternalRecordFormat:(AudioStreamBasicDescription *)asdes {
    _outputFormat = [[AVAudioFormat alloc] initWithStreamDescription:asdes];
    return YES;
}

- (BOOL)uninitializeExternalRecord {
    _outputFormat = nil;
    return YES;
}

- (BOOL)startExternalRecord {
    self.started = YES;
    return self.started;
}

- (BOOL)stopExternalRecord {
    self.started = NO;
    return self.started;
}

- (OSStatus)onRenderData:(AudioUnitRenderActionFlags *)flags
                    time:(const AudioTimeStamp *)time
                  busNum:(UInt32)busNum
               numFrames:(UInt32)numFrames
                  ioData:(AudioBufferList *)ioData {
    int ret = noErr;
    do {
        if (!self.started) {
            ret = -1;
            break;
        }
        AudioBufferList *sourceAbl = self.bufferList.abl;
        if (sourceAbl->mNumberBuffers != ioData->mNumberBuffers) {
            ret = -1;
            break;
        }
        for (int i = 0; i < sourceAbl->mNumberBuffers; i++) {
            AudioBuffer renderBuf = ioData->mBuffers[i];
            AudioBuffer sourceBuf = sourceAbl->mBuffers[i];
            if (sourceBuf.mDataByteSize != renderBuf.mDataByteSize) {
                ret = -1;
                break;
            }
            memcpy(renderBuf.mData, sourceBuf.mData, sourceBuf.mDataByteSize);
        }
    } while (NO);
    return ret;
}
@end

@interface BDCloudAVRTCRoomAudioPlayerDevice : NSObject<BaiduRtcRoomApiAudioExternalPlayerDevice, BDCloudAVRTCRoomAudioPlayerDeviceDourceClient>
{
    AudioStreamBasicDescription _playDes;
}

@property (nonatomic, weak) id<BaiduRtcRoomApiAudioExternalPlayerClient> client;
@property (nonatomic, weak) id<BDCloudAVRTCRoomAudioPlayerDeviceSource> source;
@property (atomic, assign) BOOL started;

- (instancetype)initWithClient:(id<BaiduRtcRoomApiAudioExternalPlayerClient>)client
                        source:(id<BDCloudAVRTCRoomAudioPlayerDeviceSource>)source;
@end

@implementation BDCloudAVRTCRoomAudioPlayerDevice
- (instancetype)initWithClient:(id<BaiduRtcRoomApiAudioExternalPlayerClient>)client
                        source:(id<BDCloudAVRTCRoomAudioPlayerDeviceSource>)source {
    if (self = [super init]) {
        _client = client;
        _source = source;
    }
    return self;
}

- (BOOL)initializeExternalPlayerFormat:(AudioStreamBasicDescription *)asdes {
    _playDes.mSampleRate = asdes->mSampleRate;
    _playDes.mFormatID = asdes->mFormatID;
    _playDes.mFormatFlags = asdes->mFormatFlags;
    _playDes.mChannelsPerFrame = asdes->mChannelsPerFrame;
    _playDes.mFramesPerPacket = asdes->mFramesPerPacket;
    _playDes.mBitsPerChannel = asdes->mBitsPerChannel;
    _playDes.mBytesPerFrame = asdes->mBytesPerFrame;
    _playDes.mBytesPerPacket = asdes->mBytesPerPacket;
    return YES;
}

- (BOOL)uninitializeExternalPlayer {
    return YES;
}

- (BOOL)startExternalPlayer {
    if ([self.source respondsToSelector:@selector(onAudioPlayerStart:client:)]) {
        [self.source onAudioPlayerStart:&_playDes
                                       client:self];
        self.started = YES;
        return YES;
    }
    return NO;
}

- (BOOL)stopExternalPlayer {
    if ([self.source respondsToSelector:@selector(onAudioPlayerStop)]) {
        [self.source onAudioPlayerStop];
        self.started = NO;
        return YES;
    }
    return NO;
}

#pragma mark --BDCloudAVRTCRoomAudioPlayerDeviceDourceClient
- (OSStatus)onRequestPlayoutData:(AudioUnitRenderActionFlags *)flags
                       timeStamp:(const AudioTimeStamp *)timeStamp
                       busNumber:(UInt32)busNumber
                       numFrames:(UInt32)numFrames
                            data:(AudioBufferList *)data {
    if (self.started) {
        return [self.client onGetPlayoutData:flags
                                        time:timeStamp
                                      busNum:busNumber
                                   numFrames:numFrames
                                      ioData:data];
    } else {
        return 0;
    }
}
@end

@interface BDCloudAVRTCRoomAudioDeviceFactory ()
@property (nonatomic, strong) BDCloudAVRTCRoomAudioRecordDevice *recordDevice;
@property (nonatomic, strong) BDCloudAVRTCRoomAudioPlayerDevice *playerDevice;
@property (nonatomic, weak)   id<BDCloudAVRTCRoomAudioPlayerDeviceSource> source;
@end

@implementation BDCloudAVRTCRoomAudioDeviceFactory

- (instancetype)initWithPlayerDeviceSource:(id<BDCloudAVRTCRoomAudioPlayerDeviceSource>)source {
    if (self = [super init]) {
        self.source = source;
    }
    return self;
}

- (void)frame:(CMSampleBufferRef)frame type:(NSString *)type {
    if ([type isEqualToString:AVMediaTypeAudio] && self.recordDevice) {
        [self.recordDevice onIncommingBuffer:frame];
    }
}

- (id<BaiduRtcRoomApiAudioExternalRecordDevice>)createExternalRecord:(id<BaiduRtcRoomApiAudioExternalRecordClient>)client {
    self.recordDevice = [[BDCloudAVRTCRoomAudioRecordDevice alloc] initWithClient:client];
    return self.recordDevice;
}

- (id<BaiduRtcRoomApiAudioExternalPlayerDevice>)createExternalPlayer:(id<BaiduRtcRoomApiAudioExternalPlayerClient>)client {
    self.playerDevice = [[BDCloudAVRTCRoomAudioPlayerDevice alloc] initWithClient:client
                         source:self.source];
    return self.playerDevice;
}
@end
