//
//  BDCloudAVExternalCaptureSession.m
//  svlapp
//
//  Created by Fu,Sheng(ACG VCP) on 9.11.20.
//  Copyright © 2020 fusheng. All rights reserved.
//

#import "BDCloudAVExternalCaptureSession.h"

static float audioSampleRate = 44100;

@interface BDCloudAVExternalCaptureSession ()<AVCaptureVideoDataOutputSampleBufferDelegate>

// video
@property (nonatomic, strong) AVCaptureDeviceInput     *captureDeviceInput;
@property (nonatomic, strong) AVCaptureVideoDataOutput *captureVideoDataOutput;
@property (nonatomic, strong) AVCaptureSession         *captureSession;
@property (nonatomic, strong) AVCaptureConnection      *captureConnection;
@property (nonatomic, strong) AVCaptureDevice          *camera;
@property (nonatomic, strong) AVCaptureSessionPreset   preset;
@property (nonatomic, assign) NSInteger                frameRate;

// audio
@property (nonatomic, assign) AudioComponentInstance componetInstance;
@property (nonatomic, assign) AudioComponent component;
@property (nonatomic, strong) dispatch_queue_t audioTaskQueue;
@property (nonatomic, assign) BOOL isRunning;
@property (nonatomic, copy)   BDCloudCaputureAudioPlayerCallback playerCallback;
@end

@implementation BDCloudAVExternalCaptureSession

@synthesize preView = _preView;
@synthesize enHanceVideo = _enHanceVideo;
@synthesize snapShotVideo = _snapShotVideo;

- (void)dealloc {
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [self stopPreview];
    if (self.componetInstance) {
        AudioComponentInstanceDispose(self.componetInstance);
        self.componetInstance = nil;
        self.component = nil;
    }
}

- (instancetype)initWithPreset:(AVCaptureSessionPreset)sessionPreset
                     frameRate:(NSInteger)frameRate {
    if (self = [super init]) {
        _preView = [[UIView alloc] initWithFrame:[UIScreen mainScreen].bounds];
        _preView.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
        _render = [[BDCloudAVMetalRender alloc] init];
        _render.scalingMode = BDCloudAVMetalScalingModeAspectFill;
        _render.renderView.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
        _videoInput = [[BDCloudAVOutput alloc] initWithMediaType:AVMediaTypeVideo];
        _audioInput = [[BDCloudAVOutput alloc] initWithMediaType:AVMediaTypeAudio];
        _preset = sessionPreset;
        _frameRate = frameRate;
        
        self.render.renderView.frame = self.preView.bounds;
        [self.preView addSubview:self.render.renderView];
//        [self.videoInput addTarget:self.render];
//        [self.videoInput addTarget:self.enHanceVideo];
//        [self.enHanceVideo addTarget:self.render];
        
        [self initVideoCapture:AVCaptureDevicePositionFront];
        [self initAudioCapture];
    }
    return self;
}

- (void)setAudioSession:(BOOL)active {
    if (active) {
        [[AVAudioSession sharedInstance] setCategory:AVAudioSessionCategoryPlayAndRecord
                                         withOptions:(AVAudioSessionCategoryOptionAllowBluetooth | AVAudioSessionCategoryOptionDuckOthers | AVAudioSessionCategoryOptionDefaultToSpeaker)
                                               error:nil];
        [[AVAudioSession sharedInstance] setActive:YES error:nil];
    } else {
        [[AVAudioSession sharedInstance] setActive:NO error:nil];
    }
}

- (void)setOverrideOutputAudioPort:(AVAudioSessionPortOverride)portOverride {
    [[AVAudioSession sharedInstance] overrideOutputAudioPort:portOverride
                                                       error:nil];
}

- (BDCloudEnhanceVideo *) enHanceVideo {
    if (!_enHanceVideo) {
        _enHanceVideo = [[BDCloudEnhanceVideo alloc]
                            initWithMediaType:AVMediaTypeVideo];
    }
    return  _enHanceVideo;
}
- (BDCloudSnapShot *) snapShotVideo {
    if (!_snapShotVideo) {
        _snapShotVideo = [[BDCloudSnapShot alloc]
                            initWithMediaType:AVMediaTypeVideo];
    }
    return  _snapShotVideo;
}

#pragma mark --BDCloudAVCaptureProtocol
- (BOOL)startPreview {
    if (self.isRunning) {
        return NO;
    }
    // 注意切换audiosession
    [self setAudioSession:YES];
    __weak typeof(self) wSelf = self;
    dispatch_async(self.audioTaskQueue, ^{
        wSelf.isRunning = YES;
        [wSelf.captureSession startRunning];
        AudioOutputUnitStart(wSelf.componetInstance);
    });
    return YES;
}

- (BOOL)stopPreview {
    if (!self.isRunning) {
        return NO;
    }
    // 注意切换audiosession
    [self setAudioSession:NO];
    __weak typeof(self) wSelf = self;
    dispatch_sync(self.audioTaskQueue, ^{
        wSelf.isRunning = NO;
        NSLog(@"MicrophoneSource: stopRunning");
        AudioOutputUnitStop(wSelf.componetInstance);
        [wSelf.captureSession stopRunning];
    });
    return YES;
}

- (void)startAudioPlay:(const AudioStreamBasicDescription *)playDes playerCallback:(BDCloudCaputureAudioPlayerCallback)playerCallback {
    __weak typeof(self) wSelf = self;
    dispatch_sync(self.audioTaskQueue, ^{
        if (!wSelf.isRunning) {
            return;
        }
        // 暂停音频录制
        AudioOutputUnitStop(wSelf.componetInstance);
        AudioUnitUninitialize(wSelf.componetInstance);
        
        // 开启音频播放
        UInt32 playEnable = 1;
        OSStatus ret = AudioUnitSetProperty(wSelf.componetInstance,
                                            kAudioOutputUnitProperty_EnableIO,
                                            kAudioUnitScope_Output,
                                            0,
                                            &playEnable,
                                            sizeof(playEnable));
        if (ret != noErr) {
            NSLog(@"Failed to kAudioOutputUnitProperty_EnableIO microphone!");
        }
        
        // 开启回声消除
        UInt32 aecEnable = 1;
        ret = AudioUnitSetProperty(wSelf.componetInstance,
                                   kAUVoiceIOProperty_BypassVoiceProcessing,
                                   kAudioUnitScope_Global,
                                   0,
                                   &aecEnable,
                                   sizeof(aecEnable));
        if(ret != noErr) {
            NSLog(@"Failed to kAUVoiceIOProperty_BypassVoiceProcessing microphone!");
        }
        
        // 设置音频播放参数
        ret = AudioUnitSetProperty(wSelf.componetInstance,
                                   kAudioUnitProperty_StreamFormat,
                                   kAudioUnitScope_Input,
                                   0,
                                   playDes,
                                   sizeof(AudioStreamBasicDescription));
        if(ret != noErr) {
            NSLog(@"Failed to playDes kAudioUnitProperty_StreamFormat microphone!");
        }
        
        // 设置音频回调函数
        AURenderCallbackStruct playCb;
        playCb.inputProcRefCon = (__bridge void *)(wSelf);
        playCb.inputProc = handleOutputBuffer;
        ret = AudioUnitSetProperty(wSelf.componetInstance,
                                   kAudioUnitProperty_SetRenderCallback,
                                   kAudioUnitScope_Input,
                                   0,
                                   &playCb,
                                   sizeof(playCb));
        if(ret != noErr) {
            NSLog(@"Failed to kAudioUnitProperty_SetRenderCallback microphone!");
        }
        
        ret = AudioUnitInitialize(wSelf.componetInstance);
        if(ret != noErr) {
            NSLog(@"Failed to AudioUnitInitialize microphone!");
        }
        
        ret = AudioOutputUnitStart(wSelf.componetInstance);
        if(ret != noErr) {
            NSLog(@"Failed to start microphone!");
        }
        
        wSelf.playerCallback = playerCallback;
    });
}

- (void)stopAudioPlay {
    __weak typeof(self) wSelf = self;
    dispatch_sync(self.audioTaskQueue, ^{
        if (!wSelf.isRunning) {
            return;
        }
        AudioOutputUnitStop(wSelf.componetInstance);
        AudioUnitUninitialize(wSelf.componetInstance);
        UInt32 playEnable = 0;
        OSStatus ret = AudioUnitSetProperty(wSelf.componetInstance,
                                            kAudioOutputUnitProperty_EnableIO,
                                            kAudioUnitScope_Output,
                                            0,
                                            &playEnable,
                                            sizeof(playEnable));
        if (ret != noErr) {
            NSLog(@"Failed to playEnable microphone!");
        }
        
        AudioUnitInitialize(wSelf.componetInstance);
        ret = AudioOutputUnitStart(wSelf.componetInstance);
        if(ret != noErr) {
            NSLog(@"Failed to start microphone!");
        }
    });
}

/* ------------ 所有相机相关方法请自己实现。 ----------------*/
- (void)togglelocalMirror {}
- (void)toggleTorch:(BOOL)torchOn {}
- (void)toggleFlash:(AVCaptureFlashMode)mode {}
- (void)switchCameraPosition:(AVCaptureDevicePosition)position {
    __weak typeof(self) wSelf = self;
    dispatch_sync(self.audioTaskQueue, ^{
        if (!wSelf.isRunning) {
            return;
        }
        [wSelf.captureSession stopRunning];
        [wSelf initVideoCapture:position];
        [wSelf.captureSession startRunning];
    });
}

- (void)setCameraContinuousAutofocus:(BOOL)autoFocus {}
- (void)setCameraFocusPointOfInterest:(CGPoint)point {}
- (void)setCameraExposurePointOfInterest:(CGPoint)point {}
- (void)setCameraZoomFactor:(CGFloat)zoomFactor {}
- (void)autoPauseAudioBackground:(BOOL)pause {}
- (BOOL)setImageOutput:(nullable UIImage *)image { return NO;}

//// 视频截图
- (void)startSnapShot:(BDSnapShotCompleteBlock)complete {
    if (_snapShotVideo) {
        [_snapShotVideo startSnapShot:complete];
    }
}

#pragma mark  ----- audioCapture
- (void)initAudioCapture {
    self.audioTaskQueue = dispatch_queue_create("com.baidu.svlapp.audioCapture.Queue", NULL);
    [[NSNotificationCenter defaultCenter] addObserver: self
                                             selector: @selector(handleInterruption:)
                                                 name: AVAudioSessionInterruptionNotification
                                               object: [AVAudioSession sharedInstance]];
    AudioComponentDescription acd;
    acd.componentType = kAudioUnitType_Output;
    acd.componentSubType = kAudioUnitSubType_RemoteIO;
    acd.componentManufacturer = kAudioUnitManufacturer_Apple;
    acd.componentFlags = 0;
    acd.componentFlagsMask = 0;
    
    self.component = AudioComponentFindNext(NULL, &acd);
    
    OSStatus status = noErr;
    status = AudioComponentInstanceNew(self.component, &_componetInstance);
    
    if (noErr != status) {
        return;
    }
    
    // io
    UInt32 enable = 1;
    AudioUnitSetProperty(self.componetInstance, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Input, 1, &enable, sizeof(enable));
    enable = 0;
    AudioUnitSetProperty(self.componetInstance, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Output, 0, &enable, sizeof(enable));
    
    AudioStreamBasicDescription desc = {0};
    desc.mSampleRate = audioSampleRate;
    desc.mFormatID = kAudioFormatLinearPCM;
    desc.mFormatFlags = kAudioFormatFlagIsSignedInteger | kAudioFormatFlagsNativeEndian | kAudioFormatFlagIsPacked;
    desc.mChannelsPerFrame = 2;
    desc.mFramesPerPacket = 1;
    desc.mBitsPerChannel = 16;
    desc.mBytesPerFrame = desc.mBitsPerChannel / 8 * desc.mChannelsPerFrame;
    desc.mBytesPerPacket = desc.mBytesPerFrame * desc.mFramesPerPacket;

    AURenderCallbackStruct cb;
    cb.inputProcRefCon = (__bridge void *)(self);
    cb.inputProc = handleInputBuffer;
    AudioUnitSetProperty(self.componetInstance, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Output, 1, &desc, sizeof(desc));
    AudioUnitSetProperty(self.componetInstance, kAudioOutputUnitProperty_SetInputCallback, kAudioUnitScope_Global, 1, &cb, sizeof(cb));
    
    status = AudioUnitInitialize(self.componetInstance);
    
    if (noErr != status) {
        NSLog(@"error code = %d", (int)status);
    }
}

#pragma mark ----videoCapture
- (void)initVideoCapture:(AVCaptureDevicePosition)position {
    // 获取所有摄像头
    NSArray *cameras= [AVCaptureDevice devicesWithMediaType:AVMediaTypeVideo];
    // 获取前置摄像头
    NSArray *captureDeviceArray = [cameras filteredArrayUsingPredicate:[NSPredicate predicateWithFormat:@"position == %d", position]];
    if (!captureDeviceArray.count)
    {
        NSLog(@"获取前置摄像头失败");
        return;
    }
    // 转化为输入设备
    self.camera = captureDeviceArray.firstObject;
    NSError *errorMessage = nil;
    self.captureDeviceInput = [AVCaptureDeviceInput deviceInputWithDevice:self.camera error:&errorMessage];
    if (errorMessage)
    {
        NSLog(@"AVCaptureDevice转AVCaptureDeviceInput失败");
        return;
    }
    
    // 设置视频输出
    self.captureVideoDataOutput = [[AVCaptureVideoDataOutput alloc] init];
    
    // 设置视频数据格式[NSNumber numberWithInt:kCVPixelFormatType_32BGRA],
    NSDictionary *videoSetting = [NSDictionary dictionaryWithObjectsAndKeys:
                                  [NSNumber numberWithInt:kCVPixelFormatType_420YpCbCr8BiPlanarVideoRange],
                                  kCVPixelBufferPixelFormatTypeKey,
                                  nil];
    [self.captureVideoDataOutput setVideoSettings:videoSetting];
    
    // 设置输出代理、串行队列和数据回调
    dispatch_queue_t outputQueue = dispatch_queue_create("ACVideoCaptureOutputQueue", DISPATCH_QUEUE_SERIAL);
    [self.captureVideoDataOutput setSampleBufferDelegate:self queue:outputQueue];
    // 丢弃延迟的帧
    self.captureVideoDataOutput.alwaysDiscardsLateVideoFrames = YES;
    
    self.captureSession = [[AVCaptureSession alloc] init];
    self.captureSession.usesApplicationAudioSession = NO;
    if ([self.captureSession canAddInput:self.captureDeviceInput])
    {
        [self.captureSession addInput:self.captureDeviceInput];
    }
    if ([self.captureSession canAddOutput:self.captureVideoDataOutput])
    {
        [self.captureSession addOutput:self.captureVideoDataOutput];
    }
    if ([self.captureSession canSetSessionPreset:self.preset])
    {
        self.captureSession.sessionPreset = self.preset;
    }
    
    [self.captureSession beginConfiguration];
    if ([self.camera lockForConfiguration:nil]) {
        self.camera.activeVideoMinFrameDuration = CMTimeMake(1, (int)self.frameRate);
        self.camera.activeVideoMaxFrameDuration = CMTimeMake(1, (int)self.frameRate);
        [self.camera unlockForConfiguration];
    }
    self.captureConnection = [self.captureVideoDataOutput connectionWithMediaType:AVMediaTypeVideo];
    self.captureConnection.videoOrientation = AVCaptureVideoOrientationPortrait;
    if (self.camera.position == AVCaptureDevicePositionFront && self.captureConnection.supportsVideoMirroring) {
        self.captureConnection.videoMirrored = YES;
    }
    [self.captureSession commitConfiguration];
}

- (void)captureOutput:(AVCaptureOutput *)output didOutputSampleBuffer:(CMSampleBufferRef)sampleBuffer fromConnection:(AVCaptureConnection *)connection {
    CFRetain(sampleBuffer);
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        [self.videoInput raiseFrame:sampleBuffer];
        
        CFRelease(sampleBuffer);
    });
}

- (void)handleInterruption:(NSNotification *)notification {
    NSInteger reason = 0;
    NSString *reasonStr = @"";
    if ([notification.name isEqualToString:AVAudioSessionInterruptionNotification]) {
        reason = [[[notification userInfo] objectForKey:AVAudioSessionInterruptionTypeKey] integerValue];
        if (reason == AVAudioSessionInterruptionTypeBegan) {
            if (self.isRunning) {
                dispatch_sync(self.audioTaskQueue, ^{
                    AudioOutputUnitStop(self.componetInstance);
                });
            }
        }
        
        if (reason == AVAudioSessionInterruptionTypeEnded) {
            reasonStr = @"AVAudioSessionInterruptionTypeEnded";
            NSNumber *seccondReason = [[notification userInfo] objectForKey:AVAudioSessionInterruptionOptionKey];
            switch ([seccondReason integerValue]) {
                case AVAudioSessionInterruptionOptionShouldResume:
                    if (self.isRunning) {
                        dispatch_async(self.audioTaskQueue, ^{
                            NSLog(@"MicrophoneSource: startRunning");
                            AudioOutputUnitStart(self.componetInstance);
                        });
                    }
                    break;
                default:
                    break;
            }
        }
    }
    NSLog(@"handleInterruption: %@ reason %@", [notification name], reasonStr);
}


// 外部PCM音频
- (void)onGetExternalAudioBuffer:(NSData *)data
                      sampleRate:(int32_t)sampleRate
                    channelCount:(int32_t)channelCount {
    AudioBuffer buffer;
    buffer.mData = (void *)[data bytes];
    buffer.mDataByteSize = (UInt32)[data length];
    buffer.mNumberChannels = channelCount;
    
    AudioBufferList buffers;
    buffers.mNumberBuffers = 1;
    buffers.mBuffers[0] = buffer;
    
    AudioStreamBasicDescription desc = {0};
    desc.mSampleRate = sampleRate;
    desc.mFormatID = kAudioFormatLinearPCM;
    desc.mFormatFlags = (kAudioFormatFlagIsSignedInteger | kAudioFormatFlagsNativeEndian | kAudioFormatFlagIsPacked);
    desc.mChannelsPerFrame = channelCount;
    desc.mFramesPerPacket = 1;
    desc.mBitsPerChannel = 16 ;
    desc.mBytesPerFrame = desc.mBitsPerChannel / 8 * desc.mChannelsPerFrame;
    desc.mBytesPerPacket = desc.mBytesPerFrame * desc.mFramesPerPacket;
    
    BDCloudAVAudioBufferList *abl = [BDCloudAVAudioBufferList fromASBD:&desc
                                                                  data:&buffers
                                                               samples:1024];
    if (!abl) {
        return;
    }
    BDCloudAVCMSampleBuffer *sampleBuffer = [BDCloudAVCMSampleBuffer fromAudioBufferList:abl
                                                                                     pts:kCMTimeZero];
    if (!sampleBuffer) {
        return;
    }
    [self.audioInput raiseFrame:sampleBuffer.buffer];
}


#pragma mark -- audioCallBack
static OSStatus handleInputBuffer(void *inRefCon,
                                  AudioUnitRenderActionFlags *ioActionFlags,
                                  const AudioTimeStamp *inTimeStamp,
                                  UInt32 inBusNumber,
                                  UInt32 inNumberFrames,
                                  AudioBufferList *ioData) {
    @autoreleasepool {
        BDCloudAVExternalCaptureSession *source = (__bridge BDCloudAVExternalCaptureSession *)inRefCon;
        if (!source) return -1;
        
        AudioBuffer buffer;
        buffer.mData = NULL;
        buffer.mDataByteSize = 0;
        buffer.mNumberChannels = 1;
        
        AudioBufferList buffers;
        buffers.mNumberBuffers = 1;
        buffers.mBuffers[0] = buffer;
        
        OSStatus status = AudioUnitRender(source.componetInstance,
                                          ioActionFlags,
                                          inTimeStamp,
                                          inBusNumber,
                                          inNumberFrames,
                                          &buffers);
//        {
//             外部PCM采集数据用例
//            AudioBuffer buffer1 = buffers.mBuffers[0];
//            [source onGetExternalAudioBuffer:[NSData dataWithBytes:buffer1.mData length:buffer1.mDataByteSize]
//                                  sampleRate:audioSampleRate
//                                channelCount:2];
//            return 0;
//        }
        
        if (!status) {
            AudioStreamBasicDescription desc = {0};
            desc.mSampleRate = audioSampleRate;
            desc.mFormatID = kAudioFormatLinearPCM;
            desc.mFormatFlags = (kAudioFormatFlagIsSignedInteger | kAudioFormatFlagsNativeEndian | kAudioFormatFlagIsPacked);
            desc.mChannelsPerFrame = 2;
            desc.mFramesPerPacket = 1;
            desc.mBitsPerChannel = 16 ;
            desc.mBytesPerFrame = desc.mBitsPerChannel / 8 * desc.mChannelsPerFrame;
            desc.mBytesPerPacket = desc.mBytesPerFrame * desc.mFramesPerPacket;
            
            NSTimeInterval seconds = [AVAudioTime secondsForHostTime:inTimeStamp->mHostTime];
            CMTime pts = CMTimeMake(seconds * 1000000000, 1000000000);
            CMTime duration = CMTimeMake(1, audioSampleRate);
            
            CMSampleTimingInfo timing = {duration,  pts, kCMTimeInvalid};
            
            CMFormatDescriptionRef format = NULL;
            status = CMAudioFormatDescriptionCreate(kCFAllocatorDefault, &desc, 0, NULL, 0, NULL, NULL, &format);
            if (status != noErr) {
                if(format){CFRelease(format);}
                return status;
            }
            
            CMSampleBufferRef sampleBufferRef = NULL;
            status = CMSampleBufferCreate(kCFAllocatorDefault,
                                          NULL,
                                          false,
                                          NULL,
                                          NULL,
                                          format,
                                          (CMItemCount)inNumberFrames,
                                          1,
                                          &timing,
                                          0,
                                          NULL,
                                          &sampleBufferRef);
            
            if (status != noErr) {
                if (sampleBufferRef) { CFRelease(sampleBufferRef);}
                if (format) { CFRelease(format);}
                return status;
            }
            status = CMSampleBufferSetDataBufferFromAudioBufferList(sampleBufferRef,
                                                                    kCFAllocatorDefault,
                                                                    kCFAllocatorDefault,
                                                                    0,
                                                                    &buffers);
            if (status == noErr) {
                [source.audioInput raiseFrame:sampleBufferRef];
            }
            
            if (sampleBufferRef) { CFRelease(sampleBufferRef);}
            if (format) { CFRelease(format);}
        }
        return 0;
    }
}

OSStatus handleOutputBuffer(void *inRefCon,
                            AudioUnitRenderActionFlags * ioActionFlags,
                            const AudioTimeStamp *   inTimeStamp,
                            UInt32   inBusNumber,
                            UInt32   inNumberFrames,
                            AudioBufferList * __nullable ioData) {
    BDCloudAVExternalCaptureSession *source = (__bridge BDCloudAVExternalCaptureSession *)inRefCon;
    if (source.playerCallback) {
        return source.playerCallback(ioActionFlags,
                                     inTimeStamp,
                                     inBusNumber,
                                     inNumberFrames,
                                     ioData);
    }
    return 0;
}
@end
