package main

import (
	"context"
	"errors"
	"flag"
	"fmt"
	"io"
	"log"
	"math/rand"
	"os"
	"strings"
	"time"

	"baidurtc"

	"github.com/pion/webrtc/v4/pkg/media/h264reader"
	"github.com/pion/webrtc/v4/pkg/media/h265reader"
	"github.com/pion/webrtc/v4/pkg/media/oggreader"
)

func PrepareSendMedia(c *baidurtc.BRTCClient, l *myListener, afn string, vfn string) {
    var (
        audioFileName     = afn
        videoFileName     = vfn
        oggPageDuration   = time.Millisecond * 20
        h264FrameDuration = time.Millisecond * 33
        videoFrameDuration = time.Millisecond * 40
    )
    // Assert that we have an audio or video file
    _, err := os.Stat(videoFileName)
    haveVideoFile := !os.IsNotExist(err)

    _, err = os.Stat(audioFileName)
    haveAudioFile := !os.IsNotExist(err)

    if !haveAudioFile && !haveVideoFile {
        log.Println("Could not find `" + audioFileName + "` or `" + videoFileName + "`")
        return
    }

    l.PubConnectedCtx , l.PubConnectedCtxCancel = context.WithCancel(context.Background())

    if haveVideoFile {
        if strings.HasSuffix(vfn, ".h265") {
            go func() {
                // Open a H265 file and start reading
                file, h265Err := os.Open(videoFileName)
                if h265Err != nil {
                    panic(h265Err)
                }

                h265, h265Err := h265reader.NewReader(file)
                if h265Err != nil {
                    panic(h265Err)
                }

                // Wait for connection established
                <- l.PubConnectedCtx.Done()

                ticker := time.NewTicker(videoFrameDuration)
                for ; true; <-ticker.C {
                    nal, h265Err := h265.NextNAL()
                    if errors.Is(h265Err, io.EOF) {
                        file.Seek(0, 0)
                        continue
                    }
                    if h265Err != nil {
                        continue
                    }

                    // log.Printf("SendVideo frame len: %d\n",len(nal.Data))
                    c.SendVideo(nal.Data, videoFrameDuration)
                }
            }()
        } else {
            go func() {
                // Open a H264 file and start reading using our IVFReader
                file, h264Err := os.Open(videoFileName)
                if h264Err != nil {
                    panic(h264Err)
                }

                h264, h264Err := h264reader.NewReader(file)
                if h264Err != nil {
                    panic(h264Err)
                }

                // Wait for connection established
                <- l.PubConnectedCtx.Done()

                // Send our video file frame at a time. Pace our sending so we send it at the same speed it should be played back as.
                // This isn't required since the video is timestamped, but we will such much higher loss if we send all at once.
                //
                // It is important to use a time.Ticker instead of time.Sleep because
                // * avoids accumulating skew, just calling time.Sleep didn't compensate for the time spent parsing the data
                // * works around latency issues with Sleep (see https://github.com/golang/go/issues/44343)
                ticker := time.NewTicker(h264FrameDuration)
                for ; true; <-ticker.C {
                    nal, h264Err := h264.NextNAL()
                    if errors.Is(h264Err, io.EOF) {
                        file.Seek(0, 0)
                        continue
                    }
                    if h264Err != nil {
                        continue
                    }

                    c.SendVideo(nal.Data, h264FrameDuration)
                }
            }()
        }
    }

    if haveAudioFile {
        if strings.HasSuffix(afn, ".g722") || strings.HasSuffix(afn, ".pcmu") || strings.HasSuffix(afn, ".pcma"){
            go func() {
                // Open a g722 file and start reading
                file, Err := os.Open(audioFileName)
                if Err != nil {
                    return
                }

                // Wait for connection established
                <-l.PubConnectedCtx.Done()

                fi, _ := os.Stat(audioFileName)

                payload := make([]byte, fi.Size())
                _, Err = io.ReadFull(file, payload)
                if  Err != nil {
                    return
                }
                var Pos = 0;

                ticker := time.NewTicker(time.Millisecond * 20)
                for ; true; <-ticker.C {

                    c.SendAudio(payload[Pos:Pos + 160], time.Millisecond * 20)
                    Pos += 160;

                    if Pos + 160 > int(fi.Size()) {
                        Pos = 0
                    }
                }
            }()
            return
        }

        if strings.HasSuffix(afn, ".raw") || strings.HasSuffix(afn, ".raw16k") {
            go func() {
                // Open a raw file and start reading
                file, Err := os.Open(audioFileName)
                if Err != nil {
                    return
                }

                // Wait for connection established
                <-l.PubConnectedCtx.Done()

                fi, _ := os.Stat(audioFileName)

                payload := make([]byte, fi.Size())
                _, Err = io.ReadFull(file, payload)
                if  Err != nil {
                    return
                }
                var Pos = 0;

                var Datalen = 320
                if (strings.Contains(strings.ToLower(afn), "16k")) {
                    Datalen = 640
                }

                ticker := time.NewTicker(time.Millisecond * 20)
                for ; true; <-ticker.C {
                    c.SendAudioPCM(payload[Pos:Pos + Datalen], time.Millisecond * 20)
                    Pos += Datalen;
                    if Pos + Datalen > int(fi.Size()) {
                        Pos = 0
                    }
                }
            }()
            return
        }

        if strings.HasSuffix(afn, ".opusx") {
            go func() {
                // Open a raw file and start reading
                file, Err := os.Open(audioFileName)
                if Err != nil {
                    return
                }

                // Wait for connection established
                <-l.PubConnectedCtx.Done()

                fi, _ := os.Stat(audioFileName)

                payload := make([]byte, fi.Size())
                _, Err = io.ReadFull(file, payload)
                if  Err != nil {
                    return
                }
                Pos := 0
                Datalen := 80
                Duration := time.Millisecond * 40

                ticker := time.NewTicker(Duration)
                for ; true; <-ticker.C {
                    c.SendAudio(payload[Pos:Pos + Datalen], Duration)
                    Pos += Datalen;
                    if Pos + Datalen > int(fi.Size()) {
                        Pos = 0
                    }
                }
            }()
            return
        }

        go func() {
            // Open a ogg file and start reading using our oggReader
            file, oggErr := os.Open(audioFileName)
            if oggErr != nil {
                panic(oggErr)
            }

            // Open on oggfile in non-checksum mode.
            ogg, _, oggErr := oggreader.NewWith(file)
            if oggErr != nil {
                panic(oggErr)
            }

            // Wait for connection established
            <-l.PubConnectedCtx.Done()

            // Keep track of last granule, the difference is the amount of samples in the buffer
            var lastGranule uint64

            // It is important to use a time.Ticker instead of time.Sleep because
            // * avoids accumulating skew, just calling time.Sleep didn't compensate for the time spent parsing the data
            // * works around latency issues with Sleep (see https://github.com/golang/go/issues/44343)
            ticker := time.NewTicker(oggPageDuration)
            for ; true; <-ticker.C {
                pageData, pageHeader, oggErr := ogg.ParseNextPage()
                if errors.Is(oggErr, io.EOF) {
                    ogg.ResetReader(func(bytesRead int64) io.Reader {
                        file.Seek(0, 0)
                        return file
                    })
                    continue
                }

                if oggErr != nil {
                    continue
                }

                // The amount of samples is the difference between the last and current timestamp
                sampleCount := float64(pageHeader.GranulePosition - lastGranule)
                lastGranule = pageHeader.GranulePosition
                sampleDuration := time.Duration((sampleCount/48000)*1000) * time.Millisecond

                c.SendAudio(pageData, sampleDuration)
            }
        }()
    }
}

type myListener struct {
    c *baidurtc.BRTCClient
    aw io.Writer
    vw io.Writer
    PubConnectedCtx       context.Context
    PubConnectedCtxCancel context.CancelFunc
}

func (c *myListener) OnVideoFrame(feedid uint64, video []byte, tpe string) {
    // log.Printf("OnVideoFrame feedid: %d, len: %d", feedid, len(video))
    if (c.vw != nil) {
        c.vw.Write(video)
    }
}

var lastTime = time.Now()
func (c *myListener) OnAudioData(feedid uint64, audio []byte, samlplerate int, channels int) {
    // log.Printf("OnAudioData feedid: %d, len: %d", feedid, len(audio))
    var now = time.Now();
    if (lastTime.Add(time.Microsecond * 200000).Before(now)) {
        log.Printf("OnAudioData gap too large feedid: %d, len: %d", feedid, len(audio))
    }
    lastTime = now;

    if (c.aw != nil) {
        c.aw.Write(audio)
    }
}

func (c *myListener) OnAudioPCM(feedid uint64, pcm []byte, sr int, ch int) {
    // log.Printf("OnAudioPCM feedid: %d, len: %d", feedid, len(pcm))
    if (c.aw != nil) {
        c.aw.Write(pcm)
    }
}

func (c *myListener) OnTextData(feedid uint64, txt []byte) {
    log.Printf("OnTextData feedid: %d, txt: %s", feedid, string(txt))
}

func (c *myListener) OnRTCLoginOK() {
    log.Printf("OnRTCLoginOK")
}
func (c *myListener) OnRTCLoginError() {
    log.Printf("OnRTCLoginError")
}
func (c *myListener) OnRTCConnectError() {
    log.Printf("OnRTCConnectError")
}
func (c *myListener) OnDisconnected() {
    log.Printf("OnDisconnected")
}
func (c *myListener) OnListenerNACKs(nacks uint64) {
    log.Printf("OnRTCLoginError")
}
func (c *myListener) OnNACKs(nacks uint64) {
    log.Printf("OnNACKs %d", nacks)
}
func (c *myListener) OnRTCLoginTimeout() {
    log.Printf("OnRTCLoginTimeout")
}
func (c *myListener) OnRTCWebrtcUp(feedId uint64) {
    log.Printf("OnRTCWebrtcUp %d", feedId)
}
func (c *myListener) OnRTCMediaStreamingEvent(feedId uint64, t int, sending bool) {
    log.Printf("OnRTCMediaStreamingEvent %d", feedId)
}
func (c *myListener) OnRTCHangUp(feedId uint64) {
    log.Printf("OnRTCHangUp feedId %d", feedId)
}
func (c *myListener) OnRTCFeedComing(feedId uint64, name string) {
    log.Printf("OnRTCFeedComing %d, name: %s", feedId, name)
}
func (c *myListener) OnRTCFeedLeaving(feedId uint64) {
    log.Printf("OnRTCMediaStreamingEvent %d", feedId)
}
func (c *myListener) OnRTCUserMessage(feedId uint64, msg string) {
    log.Printf("OnRTCUserMessage %d, msg: %s", feedId, msg)
}
func (c *myListener) OnRTCUserAttribute(feedId uint64, attr string) {
    log.Printf("OnRTCUserAttribute %d", feedId)
}
func (c *myListener) OnRTCUserJoinedRoom(feedId uint64, name string, attr string) {
    log.Printf("OnRTCUserJoinedRoom %d", feedId)
}
func (c *myListener) OnRTCUserLeavingRoom(feedId uint64, name string) {
    log.Printf("OnRTCUserLeavingRoom %d", feedId)
}
func (c *myListener) OnRTCError(error_code int, e string) {
    log.Printf("OnRTCError %d, %s", error_code, e)
}
func (c *myListener) OnPublisherJoined() {
    log.Printf("OnPublisherJoined")
}
func (c *myListener) OnRemoteMediaConnected(feedId uint64) {
    log.Printf("OnRemoteMediaConnected %d", feedId)
}
func (c *myListener) OnRemoteMediaDisconnected(feedId uint64) {
    log.Printf("OnRemoteMediaDisconnected %d", feedId)
}
func (c *myListener) OnRemoteMediaUnpublished(feedId uint64) {
    log.Printf("OnRemoteMediaUnpublished %d", feedId)
}
func (c *myListener) OnLocalMediaConnected() {
    log.Printf("OnLocalMediaConnected")
    if c.PubConnectedCtxCancel != nil {
        c.PubConnectedCtxCancel()
    }
}
func (c *myListener) OnLocalMediaDisconnected() {
    log.Printf("OnLocalMediaDisconnected")
}
func (c *myListener) OnLocalMediaPublishedOK() {
    log.Printf("OnLocalMediaPublishedOK")
}
func (c *myListener) OnLocalMediaUnpublished() {
    log.Printf("OnLocalMediaUnpublished")
}

var ac string
var vc string
var af string
var adump string
var ss string
var appid string
var room string
var nv string
var na string

func init() {
    flag.StringVar(&ac, "ac", "opus", "audio codec")
    flag.StringVar(&vc, "vc", "h264", "video codec")
    flag.StringVar(&af, "af", "", "audio format")
    flag.StringVar(&adump, "adump", "", "dump audio format")
    flag.StringVar(&ss, "s", "", "server url")
    flag.StringVar(&appid, "appid", "", "AppID")
    flag.StringVar(&appid, "a", "", "AppID")
    flag.StringVar(&room, "r", "", "Room Name")
    flag.StringVar(&nv, "nv", "", "Not using video")
    flag.StringVar(&na, "na", "", "Not using audio")
}

func main() {
    flag.Parse()
    log.SetFlags(log.LstdFlags | log.Lmicroseconds)

    log.Println("BRTC SDK Version is: " + baidurtc.Version())
    baidurtc.EnableLog(true)
    c := baidurtc.NewBRTCClient()

    aftype := ac
    if ac == "opus" {
        aftype = "ogg"
        c.SetAudioOutOgg(true)
    }

    outOpusxOgg := false

    if ac == "opusx" {
        ac = "opus"
        aftype = "opusx"
    } else if ac == "opuso" {
        ac = "opus"
        aftype = "opusx"
        outOpusxOgg = true
        c.SetAudioOutOgg(true)
    }

    adtype := aftype
    if adump != "" {
        adtype = adump
        c.SetAudioOutPCM(true)
    }

    if outOpusxOgg {
        adtype = "opusx.ogg"
    }

    if af != "" {
        aftype = af
    }

    vf, _ := os.Create("dump_video." + vc)
    af, _ := os.Create("dump_audio." + adtype)
    my := &myListener{c: c, vw: vf, aw:af}
    c.SetMediaObserver(my)
    c.SetEventObserver(my)

    afn := "../res/myaudio." + aftype
    vfn := "../res/myvideo." + vc

    c.SetRoomName("Go").SetDisplayName("s").SetUserId(90000000 + rand.Uint64()%100000).SetToken("no_token")
    // client.SetAutoSubscribe(false)
    // client.SetAudioCodec("pcmu")
    // .SetAsPublisher(false).SetFeedId(234)
    // client.SetMediaServerIP("110.242.70.4")
    // client.SetServerURL("ws://r.redwinner.cn:8800/brtc")
    // client.SetCandidateIP("127.0.0.1")
    // client.SetCandidatePort(8010)
    c.SetAudioCodec(ac).SetVideoCodec(vc)

    appid_env := os.Getenv("APPID_GET_FROM_BAIDU")
    if appid_env != "" {
        c.SetAppId(appid_env)
    }

    if appid != "" {
        c.SetAppId(appid)
    }
    if room != "" {
        c.SetRoomName(room)
    }
    if ss != "" {
        c.SetServerURL(ss)
    }
    if nv != "" {
        c.SetHasVideo(false)
        vfn = ""
    }
    if na != "" {
        c.SetHasAudio(false)
        afn = ""
    }

    PrepareSendMedia(c, my, afn, vfn)
 
    c.Login()

    for {
        fmt.Printf("********** videocall demo V0.1 *********\n")
        fmt.Printf("Please input: h to Hangup.\n")
        fmt.Printf("              r to ReDial.\n")
        fmt.Printf("              t to LoopTest.\n")
        fmt.Printf("              q to Quit.\n")
        var commandid string

        ret, _ := fmt.Scanf("%s", &commandid)
        fmt.Printf("scanf %d.\n", ret)

        if ret < 0 {
            time.Sleep(time.Second * 200)
            continue
        } else if ret == 0 {
            time.Sleep(time.Second * 200)
            continue
        }

        switch commandid[0] {
        case 'o':
            c.Logout()
        case '2':
            c.StopPublish()
        case '3':
            c.SubscribeStreaming(10033981)
        case '4':
            c.StopSubscribeStreaming(10033981)
        case 'm':
            c.SendMessageToUser("hello", 0)
        case 'd':
            c.SendData("data hello")
        case 'e':
            c.SendRawData([]byte("data hello"))
        case 't': // loop test, t100 = 100 times.
            {

            }
        }

        if commandid[0] == 'q' {
            go c.Logout()
            time.Sleep(1 * time.Second)
            break
        }
    }
}