package baidurtc

import (
	"errors"
	"io"
	"math/bits"
)

const (
    // Input and output formats
    G711Alaw = iota // G711Alaw G711 encoded PCM data
    G711Ulaw        // G711Ulaw G711  encoded PCM data
    G711Lpcm        // G711Lpcm 16bit signed linear data
)

// G711Decoder reads G711 PCM data and decodes it to 16bit 8000Hz LPCM
type G711Decoder struct {
    decode func([]byte) []byte // decoding function
    source io.Reader           // source data
}

// G711Encoder encodes 16bit 8000Hz LPCM data to G711 PCM or
// directly transcodes between A-law and u-law
type G711Encoder struct {
    input       int                 // input format
    encode      func([]byte) []byte // encoding function
    transcode   func([]byte) []byte // transcoding function
    destination io.Writer           // output data
}

// NewAlawG711Decoder returns a pointer to a G711Decoder that implements an io.Reader.
// It takes as input the source data Reader.
func NewAlawG711Decoder(reader io.Reader) (*G711Decoder, error) {
    if reader == nil {
        return nil, errors.New("io.Reader is nil")
    }
    r := G711Decoder{
        decode: DecodeAlaw,
        source: reader,
    }
    return &r, nil
}

// NewUlawG711Decoder returns a pointer to a G711Decoder that implements an io.Reader.
// It takes as input the source data Reader.
func NewUlawG711Decoder(reader io.Reader) (*G711Decoder, error) {
    if reader == nil {
        return nil, errors.New("io.Reader is nil")
    }
    r := G711Decoder{
        decode: DecodeUlaw,
        source: reader,
    }
    return &r, nil
}

// NewAlawG711Encoder returns a pointer to an G711Encoder that implements an io.Writer.
// It takes as input the destination data Writer and the input encoding format.
func NewAlawG711Encoder(writer io.Writer, input int) (*G711Encoder, error) {
    if writer == nil {
        return nil, errors.New("io.Writer is nil")
    }
    if input != G711Ulaw && input != G711Lpcm {
        return nil, errors.New("invalid input format")
    }
    w := G711Encoder{
        input:       input,
        encode:      EncodeAlaw,
        transcode:   Ulaw2Alaw,
        destination: writer,
    }
    return &w, nil
}

// NewUlawG711Encoder returns a pointer to an G711Encoder that implements an io.Writer.
// It takes as input the destination data Writer and the input encoding format.
func NewUlawG711Encoder(writer io.Writer, input int) (*G711Encoder, error) {
    if writer == nil {
        return nil, errors.New("io.Writer is nil")
    }
    if input != G711Alaw && input != G711Lpcm {
        return nil, errors.New("invalid input format")
    }
    w := G711Encoder{
        input:       input,
        encode:      EncodeUlaw,
        transcode:   Alaw2Ulaw,
        destination: writer,
    }
    return &w, nil
}

// Reset discards the G711Decoder state. This permits reusing a G711Decoder rather than allocating a new one.
func (r *G711Decoder) Reset(reader io.Reader) error {
    if reader == nil {
        return errors.New("io.Reader is nil")
    }
    r.source = reader
    return nil
}

// Reset discards the G711Encoder state. This permits reusing an G711Encoder rather than allocating a new one.
func (w *G711Encoder) Reset(writer io.Writer) error {
    if writer == nil {
        return errors.New("io.Writer is nil")
    }
    w.destination = writer
    return nil
}

// Read decodes G711 data. Reads up to len(p) bytes into p, returns the number
// of bytes read and any error encountered.
func (r *G711Decoder) Read(p []byte) (i int, err error) {
    if len(p) == 0 {
        return
    }
    b := make([]byte, len(p)/2)
    i, err = r.source.Read(b)
    copy(p, r.decode(b))
    i *= 2 // Report back the correct number of bytes
    return
}

// Write encodes G711 Data. Writes len(p) bytes from p to the underlying data stream,
// returns the number of bytes written from p (0 <= n <= len(p)) and any error encountered
// that caused the write to stop early.
func (w *G711Encoder) Write(p []byte) (i int, err error) {
    if len(p) == 0 {
        return
    }
    if w.input == G711Lpcm { // Encode LPCM data to G711
        i, err = w.destination.Write(w.encode(p))
        if err == nil && len(p)%2 != 0 {
            err = errors.New("odd number of LPCM bytes, incomplete frame")
        }
        i *= 2 // Report back the correct number of bytes written from p
    } else { // Trans-code
        i, err = w.destination.Write(w.transcode(p))
    }
    return
}

var (
    // A-law to LPCM conversion lookup table
    alaw2lpcm = [256]int16{
        -5504, -5248, -6016, -5760, -4480, -4224, -4992, -4736,
        -7552, -7296, -8064, -7808, -6528, -6272, -7040, -6784,
        -2752, -2624, -3008, -2880, -2240, -2112, -2496, -2368,
        -3776, -3648, -4032, -3904, -3264, -3136, -3520, -3392,
        -22016, -20992, -24064, -23040, -17920, -16896, -19968, -18944,
        -30208, -29184, -32256, -31232, -26112, -25088, -28160, -27136,
        -11008, -10496, -12032, -11520, -8960, -8448, -9984, -9472,
        -15104, -14592, -16128, -15616, -13056, -12544, -14080, -13568,
        -344, -328, -376, -360, -280, -264, -312, -296,
        -472, -456, -504, -488, -408, -392, -440, -424,
        -88, -72, -120, -104, -24, -8, -56, -40,
        -216, -200, -248, -232, -152, -136, -184, -168,
        -1376, -1312, -1504, -1440, -1120, -1056, -1248, -1184,
        -1888, -1824, -2016, -1952, -1632, -1568, -1760, -1696,
        -688, -656, -752, -720, -560, -528, -624, -592,
        -944, -912, -1008, -976, -816, -784, -880, -848,
        5504, 5248, 6016, 5760, 4480, 4224, 4992, 4736,
        7552, 7296, 8064, 7808, 6528, 6272, 7040, 6784,
        2752, 2624, 3008, 2880, 2240, 2112, 2496, 2368,
        3776, 3648, 4032, 3904, 3264, 3136, 3520, 3392,
        22016, 20992, 24064, 23040, 17920, 16896, 19968, 18944,
        30208, 29184, 32256, 31232, 26112, 25088, 28160, 27136,
        11008, 10496, 12032, 11520, 8960, 8448, 9984, 9472,
        15104, 14592, 16128, 15616, 13056, 12544, 14080, 13568,
        344, 328, 376, 360, 280, 264, 312, 296,
        472, 456, 504, 488, 408, 392, 440, 424,
        88, 72, 120, 104, 24, 8, 56, 40,
        216, 200, 248, 232, 152, 136, 184, 168,
        1376, 1312, 1504, 1440, 1120, 1056, 1248, 1184,
        1888, 1824, 2016, 1952, 1632, 1568, 1760, 1696,
        688, 656, 752, 720, 560, 528, 624, 592,
        944, 912, 1008, 976, 816, 784, 880, 848,
    }
    // A-law to u-law conversion lookup table based on the ITU-T G.711 specification
    alaw2ulaw = [256]uint8{
        41, 42, 39, 40, 45, 46, 43, 44, 33, 34, 31, 32, 37, 38, 35, 36,
        57, 58, 55, 56, 61, 62, 59, 60, 49, 50, 47, 48, 53, 54, 51, 52,
        10, 11, 8, 9, 14, 15, 12, 13, 2, 3, 0, 1, 6, 7, 4, 5,
        26, 27, 24, 25, 30, 31, 28, 29, 18, 19, 16, 17, 22, 23, 20, 21,
        98, 99, 96, 97, 102, 103, 100, 101, 93, 93, 92, 92, 95, 95, 94, 94,
        116, 118, 112, 114, 124, 126, 120, 122, 106, 107, 104, 105, 110, 111, 108, 109,
        72, 73, 70, 71, 76, 77, 74, 75, 64, 65, 63, 63, 68, 69, 66, 67,
        86, 87, 84, 85, 90, 91, 88, 89, 79, 79, 78, 78, 82, 83, 80, 81,
        169, 170, 167, 168, 173, 174, 171, 172, 161, 162, 159, 160, 165, 166, 163, 164,
        185, 186, 183, 184, 189, 190, 187, 188, 177, 178, 175, 176, 181, 182, 179, 180,
        138, 139, 136, 137, 142, 143, 140, 141, 130, 131, 128, 129, 134, 135, 132, 133,
        154, 155, 152, 153, 158, 159, 156, 157, 146, 147, 144, 145, 150, 151, 148, 149,
        226, 227, 224, 225, 230, 231, 228, 229, 221, 221, 220, 220, 223, 223, 222, 222,
        244, 246, 240, 242, 252, 254, 248, 250, 234, 235, 232, 233, 238, 239, 236, 237,
        200, 201, 198, 199, 204, 205, 202, 203, 192, 193, 191, 191, 196, 197, 194, 195,
        214, 215, 212, 213, 218, 219, 216, 217, 207, 207, 206, 206, 210, 211, 208, 209,
    }
)

// EncodeAlaw encodes 16bit LPCM data to G711 A-law PCM
func EncodeAlaw(lpcm []byte) []byte {
    if len(lpcm) < 2 {
        return []byte{}
    }
    alaw := make([]byte, len(lpcm)/2)
    for i, j := 0, 0; j <= len(lpcm)-2; i, j = i+1, j+2 {
        alaw[i] = EncodeAlawFrame(int16(lpcm[j]) | int16(lpcm[j+1])<<8)
    }
    return alaw
}

// EncodeAlawFrame encodes a 16bit LPCM frame to G711 A-law PCM
func EncodeAlawFrame(frame int16) uint8 {
    var compressedByte, seg, sign int16
    sign = ((^frame) >> 8) & 0x80
    if sign == 0 {
        frame = ^frame
    }
    compressedByte = frame >> 4
    if compressedByte > 15 {
        seg = int16(12 - bits.LeadingZeros16(uint16(compressedByte)))
        compressedByte >>= seg - 1
        compressedByte -= 16
        compressedByte += seg << 4
    }
    return uint8((sign | compressedByte) ^ 0x0055)
}

// DecodeAlaw decodes A-law PCM data to 16bit LPCM
func DecodeAlaw(pcm []byte) []byte {
    lpcm := make([]byte, len(pcm)*2)
    for i, j := 0, 0; i < len(pcm); i, j = i+1, j+2 {
        frame := alaw2lpcm[pcm[i]]
        lpcm[j] = byte(frame)
        lpcm[j+1] = byte(frame >> 8)
    }
    return lpcm
}

// DecodeAlawFrame decodes an A-law PCM frame to 16bit LPCM
func DecodeAlawFrame(frame uint8) int16 {
    return alaw2lpcm[frame]
}

// Alaw2Ulaw performs direct A-law to u-law data conversion
func Alaw2Ulaw(alaw []byte) []byte {
    ulaw := make([]byte, len(alaw))
    for i := 0; i < len(alaw); i++ {
        ulaw[i] = alaw2ulaw[alaw[i]]
    }
    return ulaw
}

// Alaw2UlawFrame directly converts an A-law frame to u-law
func Alaw2UlawFrame(frame uint8) uint8 {
    return alaw2ulaw[frame]
}

const (
    ulawBias = 33
    ulawClip = 0x1FFF
)

var (
    // u-law to LPCM conversion lookup table
    ulaw2lpcm = [256]int16{
        -32124, -31100, -30076, -29052, -28028, -27004, -25980, -24956,
        -23932, -22908, -21884, -20860, -19836, -18812, -17788, -16764,
        -15996, -15484, -14972, -14460, -13948, -13436, -12924, -12412,
        -11900, -11388, -10876, -10364, -9852, -9340, -8828, -8316,
        -7932, -7676, -7420, -7164, -6908, -6652, -6396, -6140,
        -5884, -5628, -5372, -5116, -4860, -4604, -4348, -4092,
        -3900, -3772, -3644, -3516, -3388, -3260, -3132, -3004,
        -2876, -2748, -2620, -2492, -2364, -2236, -2108, -1980,
        -1884, -1820, -1756, -1692, -1628, -1564, -1500, -1436,
        -1372, -1308, -1244, -1180, -1116, -1052, -988, -924,
        -876, -844, -812, -780, -748, -716, -684, -652,
        -620, -588, -556, -524, -492, -460, -428, -396,
        -372, -356, -340, -324, -308, -292, -276, -260,
        -244, -228, -212, -196, -180, -164, -148, -132,
        -120, -112, -104, -96, -88, -80, -72, -64,
        -56, -48, -40, -32, -24, -16, -8, 0,
        32124, 31100, 30076, 29052, 28028, 27004, 25980, 24956,
        23932, 22908, 21884, 20860, 19836, 18812, 17788, 16764,
        15996, 15484, 14972, 14460, 13948, 13436, 12924, 12412,
        11900, 11388, 10876, 10364, 9852, 9340, 8828, 8316,
        7932, 7676, 7420, 7164, 6908, 6652, 6396, 6140,
        5884, 5628, 5372, 5116, 4860, 4604, 4348, 4092,
        3900, 3772, 3644, 3516, 3388, 3260, 3132, 3004,
        2876, 2748, 2620, 2492, 2364, 2236, 2108, 1980,
        1884, 1820, 1756, 1692, 1628, 1564, 1500, 1436,
        1372, 1308, 1244, 1180, 1116, 1052, 988, 924,
        876, 844, 812, 780, 748, 716, 684, 652,
        620, 588, 556, 524, 492, 460, 428, 396,
        372, 356, 340, 324, 308, 292, 276, 260,
        244, 228, 212, 196, 180, 164, 148, 132,
        120, 112, 104, 96, 88, 80, 72, 64,
        56, 48, 40, 32, 24, 16, 8, 0,
    }
    // u-law to A-law conversion lookup table based on the ITU-T G.711 specification
    ulaw2alaw = [256]uint8{
        42, 43, 40, 41, 46, 47, 44, 45, 34, 35, 32, 33, 38, 39, 36, 37,
        58, 59, 56, 57, 62, 63, 60, 61, 50, 51, 48, 49, 54, 55, 52, 53,
        11, 8, 9, 14, 15, 12, 13, 2, 3, 0, 1, 6, 7, 4, 5, 26,
        27, 24, 25, 30, 31, 28, 29, 18, 19, 16, 17, 22, 23, 20, 21, 107,
        104, 105, 110, 111, 108, 109, 98, 99, 96, 97, 102, 103, 100, 101, 123, 121,
        126, 127, 124, 125, 114, 115, 112, 113, 118, 119, 116, 117, 75, 73, 79, 77,
        66, 67, 64, 65, 70, 71, 68, 69, 90, 91, 88, 89, 94, 95, 92, 93,
        82, 83, 83, 80, 80, 81, 81, 86, 86, 87, 87, 84, 84, 85, 85, 213,
        170, 171, 168, 169, 174, 175, 172, 173, 162, 163, 160, 161, 166, 167, 164, 165,
        186, 187, 184, 185, 190, 191, 188, 189, 178, 179, 176, 177, 182, 183, 180, 181,
        139, 136, 137, 142, 143, 140, 141, 130, 131, 128, 129, 134, 135, 132, 133, 154,
        155, 152, 153, 158, 159, 156, 157, 146, 147, 144, 145, 150, 151, 148, 149, 235,
        232, 233, 238, 239, 236, 237, 226, 227, 224, 225, 230, 231, 228, 229, 251, 249,
        254, 255, 252, 253, 242, 243, 240, 241, 246, 247, 244, 245, 203, 201, 207, 205,
        194, 195, 192, 193, 198, 199, 196, 197, 218, 219, 216, 217, 222, 223, 220, 221,
        210, 210, 211, 211, 208, 208, 209, 209, 214, 214, 215, 215, 212, 212, 213, 213,
    }
)

// EncodeUlaw encodes 16bit LPCM data to G711 u-law PCM
func EncodeUlaw(lpcm []byte) []byte {
    if len(lpcm) < 2 {
        return []byte{}
    }
    ulaw := make([]byte, len(lpcm)/2)
    for i, j := 0, 0; j <= len(lpcm)-2; i, j = i+1, j+2 {
        ulaw[i] = EncodeUlawFrame(int16(lpcm[j]) | int16(lpcm[j+1])<<8)
    }
    return ulaw
}

// EncodeUlawFrame encodes a 16bit LPCM frame to G711 u-law PCM
func EncodeUlawFrame(frame int16) uint8 {
    var lowNibble, seg, sign int16
    sign = ((^frame) >> 8) & 0x80
    if sign == 0 {
        frame = ^frame
    }
    frame = (frame >> 2) + ulawBias
    if frame > ulawClip {
        frame = ulawClip
    }
    seg = int16(16 - bits.LeadingZeros16(uint16(frame>>5)))
    lowNibble = 0x000F - ((frame >> (seg)) & 0x000F)
    return uint8(sign | ((8 - seg) << 4) | lowNibble)
}

// DecodeUlaw decodes u-law PCM data to 16bit LPCM
func DecodeUlaw(pcm []byte) []byte {
    lpcm := make([]byte, len(pcm)*2)
    for i, j := 0, 0; i < len(pcm); i, j = i+1, j+2 {
        frame := ulaw2lpcm[pcm[i]]
        lpcm[j] = byte(frame)
        lpcm[j+1] = byte(frame >> 8)
    }
    return lpcm
}

// DecodeUlawFrame decodes a u-law PCM frame to 16bit LPCM
func DecodeUlawFrame(frame uint8) int16 {
    return ulaw2lpcm[frame]
}

// Ulaw2Alaw performs direct u-law to A-law data conversion
func Ulaw2Alaw(ulaw []byte) []byte {
    alaw := make([]byte, len(ulaw))
    for i := 0; i < len(alaw); i++ {
        alaw[i] = ulaw2alaw[ulaw[i]]
    }
    return alaw
}

// Ulaw2AlawFrame directly converts a u-law frame to A-law
func Ulaw2AlawFrame(frame uint8) uint8 {
    return ulaw2alaw[frame]
}
