// Copyright 2024 Baidu.com All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package baidurtc

import (
	"bytes"
	"encoding/json"
	"fmt"
	"log"
	"math/rand"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"sync"
	"time"

	"github.com/gorilla/websocket"
)

var sdkVersion string = "V1.0.3"

type Transaction struct {
    onSuccess func(m map[string]interface{})
    onError   func(m map[string]interface{})
}

type RtcHandle struct {
    handleId     uint64
    feedId       uint64
    OnCreated    func(h RtcHandle)
    OnJoined     func(h RtcHandle)
    OnLeaving    func(h RtcHandle)
    OnError      func(h RtcHandle)
    OnRemoteJsep func(h RtcHandle, sdp string, typ string)
}

type RTCRemoteUser struct {
    display string
    attr    string
}

type IBRTCConnectionObserver interface {
    OnDisconnected()
    OnPublisherJoined(handleId uint64)
    OnPublisherRemoteJsep(handleId uint64, sdp string, typ string)
    OnSubscriberHandleRemoteJsep(handleId uint64,
        feedId uint64, sdp string, typ string)
    OnListenerNACKs(nacks uint64)
    OnNACKs(nacks uint64)
    OnUnpublished(feedId uint64, handleId uint64)

    OnRTCLoginOK()
    OnRTCLoginTimeout()
    OnRTCLoginError()
    OnRTCConnectError()
    OnRTCWebrtcUp(handleId uint64)
    OnRTCMediaStreamingEvent(handleId uint64, t int, sending bool)
    OnRTCHangUp(handleId uint64)
    OnRTCFeedComing(feedId uint64, name string)
    OnRTCFeedLeaving(feedId uint64)
    OnRTCUserMessage(feedId uint64, msg string)
    OnRTCUserAttribute(feedId uint64, attr string)
    OnRTCUserJoinedRoom(feedId uint64, name string, attr string)
    OnRTCUserLeavingRoom(feedId uint64, name string)
    OnRTCError(error_code int, e string)
}

type BRTCConnection struct {
    sessionId uint64
    mAliving   bool

    mUserId       uint64
    mRoomName     string
    mAppId        string
    mToken        string
    mDisplayName  string
    mOriginURI    string
    mConnetion    *websocket.Conn
    mRTCConnected bool
    mSDKTag       string
    mFullURL      string
    mediaServerIP string
    candidateIP   string
    candidatePort uint64

    roomId uint64
    feedId uint64

    asPublisher    bool
    asListener     bool
    FeedIdIsOnline bool

    HasAudio         bool
    HasVideo         bool
    HasData          bool
    AutoPublish      bool
    AutoSubscribe    bool
    VideoCodec       string
    AudioCodec       string
    LiveStreamingURL string

    brtcVersion string // brtc_v0 default as old janus, brtc_v1 for signalserver ARCH, brtc_v2 for multistream.

    publisherHandlelId uint64
    publisherSDP       string

    RemoteComingUsers map[uint64]RTCRemoteUser
    TransactionMap    map[string]Transaction
    RtcHandleMap      map[uint64]RtcHandle
    FeedsHandleMap    map[uint64]uint64

    Observer IBRTCConnectionObserver
    Wmutex    sync.Mutex

    keepAliveTask chan struct{}
}

func UInt64(n json.Number) (uint64, error) {
    return strconv.ParseUint(string(n), 10, 64)
}

func generateString(len int) (ret string) {
    r := []byte("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789")
    for cnt := 0; cnt < len-1; cnt++ {
        ret += string(r[rand.Intn(62)])
    }
    return ret
}

func NewBRTCConnection() (ret *BRTCConnection) {
    return &BRTCConnection{
        mOriginURI:        "wss://rtc.exp.bcelive.com/janus",
        TransactionMap:    make(map[string]Transaction, 0),
        RtcHandleMap:      make(map[uint64]RtcHandle, 0),
        FeedsHandleMap:    make(map[uint64]uint64, 0),
        RemoteComingUsers: make(map[uint64]RTCRemoteUser, 0),
        brtcVersion:       "brtc_v0",
        sessionId:         0,
        mSDKTag:           "BRTC.Go.SDK " + sdkVersion,
        AudioCodec:        "opus",
        VideoCodec:        "H264",
        asPublisher:       true,
        asListener:        true,
        HasAudio:          true,
        HasVideo:          true,
        HasData:           true,
        AutoPublish:       true,
        AutoSubscribe:     true,
        feedId:            0,
        Observer:          nil,
        mediaServerIP:     "",
        candidateIP:       "",
        candidatePort:     0,
        keepAliveTask:     make(chan struct{}),
    }
}

func Version() (ret string) {
    return sdkVersion
}

var enbleLogFlag = true

func EnableLog(b bool) {
    enbleLogFlag = b
}

func BRTCLog(format string, v ...any) {
    if enbleLogFlag {
        log.Printf(format, v...)
    }
}

func (c *BRTCConnection) Login() {

    dialer := websocket.Dialer{}
    header := http.Header{
        "Sec-Websocket-Protocol": []string{"janus-protocol"},
    }

    c.mFullURL = fmt.Sprintf("%s?appid=%s&roomname=%s&uid=%d&token=%s&compulsive=true",
        c.mOriginURI, c.mAppId, url.QueryEscape(c.mRoomName), c.mUserId, c.mToken)

    log.Printf("BRTCConnection.mFullURL: %s\n", c.mFullURL)

    if c.mAppId == "" || c.mRoomName == "" {
        log.Println("appid or roomname is empty")
        if c.Observer != nil {
            c.Observer.OnRTCLoginError()
        }
        return
    }

    mConnetion, _, err := dialer.Dial(c.mFullURL, header)
    if nil != err {
        log.Println(err)
        if c.Observer != nil {
            c.Observer.OnRTCLoginError()
        }
        return
    }
    c.mConnetion = mConnetion
    go onWSMessage(mConnetion, c)
}

func (s *BRTCConnection) Logout() {
    s.destroy()
    if s.mRTCConnected && s.mConnetion != nil {
        s.mConnetion.WriteControl(websocket.CloseMessage, []byte("\x03\xE8close"), time.Now().Add(30 * time.Millisecond))
        s.mConnetion.Close()
    }
}

func onWSMessage(connect *websocket.Conn, c *BRTCConnection) {
    c.onRTCOpen()
    for {
        messageType, messageData, err := connect.ReadMessage()
        if nil != err {
            log.Println(err)
            if c.keepAliveTask != nil {
                close(c.keepAliveTask)
            }
            c.mRTCConnected = false
            if c.Observer != nil {
                c.Observer.OnDisconnected()
            }
            break
        }
        switch messageType {
        case websocket.TextMessage:
            c.onRTCMessage(messageData)
        case websocket.BinaryMessage:
            fmt.Println(messageData)
        case websocket.CloseMessage:
            c.onRTCClosed()
        case websocket.PingMessage:
        case websocket.PongMessage:
        default:
        }
    }
}

func (c *BRTCConnection) SetObserver(o IBRTCConnectionObserver) (ret *BRTCConnection) {
    c.Observer = o
    return c
}

func (c *BRTCConnection) SetAppId(a string) (ret *BRTCConnection) {
    c.mAppId = a
    return c
}

func (c *BRTCConnection) SetRoomName(r string) (ret *BRTCConnection) {
    c.mRoomName = r
    return c
}

func (c *BRTCConnection) SetUserId(u uint64) (ret *BRTCConnection) {
    c.mUserId = u
    return c
}

func (c *BRTCConnection) SetDisplayName(d string) (ret *BRTCConnection) {
    c.mDisplayName = d
    return c
}

func (c *BRTCConnection) SetToken(t string) (ret *BRTCConnection) {
    c.mToken = t
    return c
}

func (c *BRTCConnection) setSDKTag(tag string) (ret *BRTCConnection) {
    c.mSDKTag = tag
    return c
}

func (c *BRTCConnection) SetAudioCodec(ac string) (ret *BRTCConnection) {
    c.AudioCodec = ac
    return c
}

func (c *BRTCConnection) SetVideoCodec(vc string) (ret *BRTCConnection) {
    c.VideoCodec = vc
    return c
}

func (c *BRTCConnection) SetAsPublisher(b bool) (ret *BRTCConnection) {
    c.asPublisher = b
    return c
}

func (c *BRTCConnection) SetAsListener(b bool) (ret *BRTCConnection) {
    c.asListener = b
    return c
}

func (c *BRTCConnection) SetHasAudio(b bool) (ret *BRTCConnection) {
    c.HasAudio = b
    return c
}

func (c *BRTCConnection) SetHasVideo(b bool) (ret *BRTCConnection) {
    c.HasVideo = b
    return c
}

func (c *BRTCConnection) SetHasData(b bool) (ret *BRTCConnection) {
    c.HasData = b
    return c
}

func (c *BRTCConnection) SetFeedId(i uint64) (ret *BRTCConnection) {
    c.feedId = i
    return c
}

func (c *BRTCConnection) SetAutoPublish(b bool) (ret *BRTCConnection) {
    c.AutoPublish = b
    return c
}

func (c *BRTCConnection) SetPublisherSDP(sdp string) (ret *BRTCConnection) {
    c.publisherSDP = sdp
    return c
}

func (c *BRTCConnection) SetAutoSubscribe(b bool) (ret *BRTCConnection) {
    c.AutoSubscribe = b
    return c
}

func (c *BRTCConnection) SetMediaServerIP(ip string) (ret *BRTCConnection) {
    c.mediaServerIP = ip
    return c
}

func (c *BRTCConnection) SetServerURL(s string) (ret *BRTCConnection) {
    c.mOriginURI = s
    return c
}

func (c *BRTCConnection) SetCandidateIP(ip string) (ret *BRTCConnection) {
    c.candidateIP = ip
    return c
}

func (c *BRTCConnection) SetCandidatePort(p uint64) (ret *BRTCConnection) {
    c.candidatePort = p
    return c
}

func (c *BRTCConnection) onRTCClosed() {
    BRTCLog("onRTCClosed.\n")
    // conn->mObserver->onRTCConnectError(conn->mObserver->handle);
    // conn->mObserver->onCirrusDisconnected(conn->mObserver->handle);
}

func (c *BRTCConnection) onRTCFailure() {
    // conn->mbQuitWatchDog = true;
    // conn->mObserver->onRTCLoginError(conn->mObserver->handle);
}

func (c *BRTCConnection) onRTCOpen() {
    c.mRTCConnected = true
    c.mAliving = true

    if c.Observer != nil {
        c.Observer.OnRTCLoginOK()
    }

    c.createSession()
}

func (c *BRTCConnection) onRTCMessage(msg []byte) {
    BRTCLog("rx: %s\n", msg)

    var dat map[string]interface{}

    d := json.NewDecoder(bytes.NewBuffer(msg))
    d.UseNumber()

    err := d.Decode(&dat)
    if err != nil {
        fmt.Println(err)
    }

    switch dat["janus"] {
    case "success":

        t := dat["transaction"].(string)

        th, hasTh := c.TransactionMap[t]
        if hasTh {
            th.onSuccess(dat)
            delete(c.TransactionMap, t)
        }
    case "error":

        t, hasTrans := dat["transaction"].(string)
        if hasTrans {
            th := c.TransactionMap[t]
            th.onError(dat)
            delete(c.TransactionMap, t)
        } else { // janus error event, disbanded, kickout, etc.
            c.onErrorEvent(dat)
        }
    case "event":
        c.onEvent(dat)
    case "ack":
        c.mAliving = true
    case "sessionevent":
        c.onSessionEvent(dat)
    case "webrtcup":
        if c.Observer != nil {
            ihandleId, hasSender := dat["sender"].(json.Number)
            var handleId uint64
            if hasSender {
                handleId, _ = UInt64(ihandleId)
            }

            c.Observer.OnRTCWebrtcUp(handleId)
        }
    case "media":
        if c.Observer != nil {
            ihandleId, hasSender := dat["sender"].(json.Number)
            var handleId uint64
            if hasSender {
                handleId, _ = UInt64(ihandleId)
            }

            r, hasR := dat["receiving"].(bool)
            if hasR {
                mT, hasT := dat["type"].(string)
                var t = 1
                if hasT {
                    if mT == "video" {
                        t = 1
                    } else {
                        t = 0
                    }

                    c.Observer.OnRTCMediaStreamingEvent(handleId, t, r)
                }
            }
        }
    case "hangup":
        if c.Observer != nil {
            ihandleId, hasSender := dat["sender"].(json.Number)
            var handleId uint64
            if hasSender {
                handleId, _ = UInt64(ihandleId)
            }

            c.Observer.OnRTCHangUp(handleId)
        }
    case "slowlink":
        if c.Observer != nil {
            i, hasN := dat["nacks"].(json.Number)
            var nacks uint64
            if hasN {
                nacks, _ = UInt64(i)
            }
            c.Observer.OnNACKs(nacks)
        }
    default:
    }
}

func (c *BRTCConnection) sendMessage(buf string) {
    if !c.mRTCConnected {
        return
    }
    BRTCLog("tx: %s\n", buf)

    c.Wmutex.Lock()
    c.mConnetion.WriteMessage(websocket.TextMessage, []byte(buf))
    c.Wmutex.Unlock()
}

func (c *BRTCConnection) createSession() {
    t := generateString(12)

    c.TransactionMap[t] = Transaction{
        onSuccess: func(m map[string]interface{}) {
            data, hasData := m["data"].(map[string]interface{})
            if hasData {
                n, _ := UInt64(data["id"].(json.Number))
                c.sessionId = n
                v, hasv := data["version"].(string)
                if hasv {
                    c.brtcVersion = v
                }

                go doKeepAlive(c)

                c.publisherCreateHandle()
            }
        },
        onError: func(m map[string]interface{}) {
        },
    }

    janusIp := ""
    if len(c.mediaServerIP) > 0 {
        janusIp = fmt.Sprintf(`"janusIp":"%s",`, c.mediaServerIP)
    }

    c.sendMessage(fmt.Sprintf(`{"janus":"create",`+
        `"transaction":"%s","sdktag":"%s",%s"uri":"%s",`+
        `"userevent":true,"sessionevent":true}`, t, c.mSDKTag, janusIp, c.mOriginURI))
}

func doKeepAlive(c *BRTCConnection) {

    ticker := time.NewTicker(25 * time.Second)
    defer ticker.Stop()

    for {
        select {
        case <-c.keepAliveTask:
            return
        case <-ticker.C:
            if (!c.mRTCConnected) {
                return
            }
            c.keepAlive()
        }
    }
}

func (c *BRTCConnection) keepAlive() {
    if c.sessionId == 0 {
        return
    }
    t := generateString(12)
    c.sendMessage(fmt.Sprintf(`{"janus":"keepalive","session_id": %d,"transaction":"%s"}`, c.sessionId, t))
}

const g_rtc_attach_cmd string = string(`{"janus":"attach","plugin":"janus.plugin.videoroom","session_id":%d,"transaction":"%s"}`)

func (c *BRTCConnection) publisherCreateHandle() {
    t := generateString(12)

    c.TransactionMap[t] = Transaction{
        onSuccess: func(m map[string]interface{}) {
            data, hasData := m["data"].(map[string]interface{})
            if hasData {
                n, _ := UInt64(data["id"].(json.Number))

                h := RtcHandle{}
                h.feedId = c.mUserId
                h.handleId = n

                h.OnRemoteJsep = func(h RtcHandle, sdp, typ string) {
                    if c.Observer != nil {
                        c.Observer.OnPublisherRemoteJsep(h.handleId, sdp, typ)
                    }
                }

                h.OnJoined = func(h RtcHandle) {
                    if c.Observer != nil {
                        c.Observer.OnPublisherJoined(h.handleId)
                    }

                    if c.AutoPublish && len(c.publisherSDP) > 5 {
                        c.StartPublish(c.publisherSDP)
                    }
                }

                c.RtcHandleMap[h.handleId] = h

                c.publisherHandlelId = n
                c.createRoom(n)
            }
        },
        onError: func(m map[string]interface{}) {
        },
    }

    c.sendMessage(fmt.Sprintf(g_rtc_attach_cmd, c.sessionId, t))
}

func (c *BRTCConnection) subscriberCreateHandle(feedId uint64, display string) {

    _, hasSubed := c.FeedsHandleMap[feedId]
    if hasSubed {
        BRTCLog("subscriberCreateHandle hasSubed\n")
        return
    }

    t := generateString(12)

    c.TransactionMap[t] = Transaction{
        onSuccess: func(m map[string]interface{}) {
            data, hasData := m["data"].(map[string]interface{})
            if hasData {
                n, _ := UInt64(data["id"].(json.Number))

                h := RtcHandle{}
                h.feedId = feedId
                h.handleId = n
                h.OnRemoteJsep = func(h RtcHandle, sdp, typ string) {
                    if c.Observer != nil {
                        c.Observer.OnSubscriberHandleRemoteJsep(h.handleId, feedId, sdp, typ)
                    }
                }
                c.RtcHandleMap[h.handleId] = h
                c.FeedsHandleMap[feedId] = h.handleId
                c.subscriberJoinRoom(h.handleId, feedId)
            }
        },
        onError: func(m map[string]interface{}) {
        },
    }

    c.sendMessage(fmt.Sprintf(g_rtc_attach_cmd, c.sessionId, t))
}

const g_rtc_create_room string = string(`{"janus":"message","body":{"request":"create","id":%d,"app_id":"%s",` +
    `"room_name":%s,"description":"%s","publishers":10000,"is_private":false,` +
    `"videocodec":"%s","audiocodec":"%s"},` +
    `"transaction":"%s",` +
    `"session_id": %d,"handle_id":%d}`)

func (c *BRTCConnection) createRoom(handleId uint64) {
    t := generateString(12)

    c.TransactionMap[t] = Transaction{
        onSuccess: func(m map[string]interface{}) {
            _, hasData := m["sender"].(json.Number)
            if hasData {
                pli, hasPli := m["plugindata"].(map[string]interface{})
                if hasPli {
                    d, hasd := pli["data"].(map[string]interface{})
                    if hasd {
                        RoomId, _ := UInt64(d["room"].(json.Number))
                        c.roomId = RoomId
                    }
                }

                if c.asPublisher {
                    c.publisherJoinRoom(handleId)
                } else if c.asListener {
                    if c.feedId != 0 {
                        c.subscriberJoinRoom(handleId, c.feedId)
                    }
                }
            }
        },
        onError: func(m map[string]interface{}) {
        },
    }

    roomname_escaped, _ := json.Marshal(c.mRoomName)
    c.sendMessage(fmt.Sprintf(g_rtc_create_room, c.mUserId, c.mAppId, roomname_escaped,
        "go", c.VideoCodec, c.AudioCodec, t, c.sessionId, handleId))
}

const g_rtc_join_room string = string(`{"janus":"message","body":{"request":"join","room":%d,"ptype":"publisher",` +
    `"display":%s,"id":%d,"app_id":"%s","room_name":%s,` +
    `"role":"publisher","token":"%s"},` +
    `"transaction":"%s","session_id":%d,"handle_id":%d}`)

func (c *BRTCConnection) publisherJoinRoom(handleId uint64) {
    t := generateString(12)

    c.TransactionMap[t] = Transaction{
        onSuccess: func(m map[string]interface{}) {
            data, hasData := m["data"].(map[string]interface{})
            if hasData {
                n, _ := UInt64(data["id"].(json.Number))
                c.publisherHandlelId = n
                c.createRoom(n)
            }
        },
        onError: func(m map[string]interface{}) {
        },
    }

    roomname_escaped, _ := json.Marshal(c.mRoomName)
    display_escaped, _ := json.Marshal(c.mDisplayName)
    c.sendMessage(fmt.Sprintf(g_rtc_join_room, c.roomId, display_escaped, c.mUserId, c.mAppId,
        roomname_escaped, c.mToken, t, c.sessionId, handleId))
}

const g_rtc_subscriber_join_room string = string(`{"janus":"message","body":{"request":"join",` +
    `"room":%d,"ptype":"listener","id":%d,"app_id":"%s",` +
    `"room_name":%s,"role":"%s","token":"%s","feed":%d},` +
    `"transaction":"%s","session_id":%d,"handle_id":%d}`)

func (c *BRTCConnection) subscriberJoinRoom(handleId uint64, feedId uint64) {
    t := generateString(12)

    c.TransactionMap[t] = Transaction{
        onSuccess: func(m map[string]interface{}) {
            data, hasData := m["data"].(map[string]interface{})
            if hasData {
                n, _ := UInt64(data["id"].(json.Number))
                c.createRoom(n)
            }
        },
        onError: func(m map[string]interface{}) {
        },
    }

    role := "publisher"

    if c.asPublisher {
        role = "publisher"
    } else {
        role = "listener"
    }

    roomname_escaped, _ := json.Marshal(c.mRoomName)
    c.sendMessage(fmt.Sprintf(g_rtc_subscriber_join_room, c.roomId, c.mUserId, c.mAppId,
        roomname_escaped, role, c.mToken, feedId, t, c.sessionId, handleId))
}

const g_rtc_offer_sdp string = string(`{"janus":"message","body":{"request":"configure",` +
    `"audio":%t,"video":%t},"jsep":{"type":"OFFER","sdp":"%s"},` +
    `"transaction":"%s","session_id":%d,"handle_id":%d}`)

func (c *BRTCConnection) publisherCreateOffer(handleId uint64, Desc string) {
    t := generateString(12)
    c.sendMessage(fmt.Sprintf(g_rtc_offer_sdp, c.HasAudio, c.HasVideo,
        Desc, t, c.sessionId, handleId))
}

func (c *BRTCConnection) StartPublish(sdp string) {
    c.publisherCreateOffer(c.publisherHandlelId, sdp)
}

const g_rtc_trickle_cmd string = string(`{"janus":"trickle","candidate":{"candidate":"%s",` +
    `"sdpMid":"0","sdpMLineIndex":0},"transaction":"%s","session_id":%d,"handle_id":%d}`)

func (c *BRTCConnection) TrickleCandicate(handleId uint64, Candidate string) {
    if !(c.brtcVersion == "brtc_v0") {
        return // It has no need in BRTC after 2020.4.14
    }
    t := generateString(12)
    c.sendMessage(fmt.Sprintf(g_rtc_trickle_cmd, Candidate, t, c.sessionId, handleId))
}

const g_rtc_trickle_complete_cmd string = string(`{"janus":"trickle","candidate":{"completed":true},` +
    `"transaction":"%s","session_id":%d,"handle_id":%d}`)

func (c *BRTCConnection) TrickleCandicateComplete(handleId uint64) {
    if !(c.brtcVersion == "brtc_v0") {
        return // It has no need in BRTC after 2020.4.14
    }
    t := generateString(12)
    c.sendMessage(fmt.Sprintf(g_rtc_trickle_complete_cmd, t, c.sessionId, handleId))
}

const g_rtc_sendmsg_cmd string = string(`{"janus":"message","transaction":"%s","body":{"request":"senddata",` +
    `"room":%d,"id":%d,"to":%d,"data":%s,"internal":%t},"session_id":%d}`)

func (c *BRTCConnection) SendMessageToUser(msg string, id uint64) {
    c.sendMessageToUserEx(msg, id, false)
}

func (c *BRTCConnection) sendMessageToUserEx(msg string, id uint64, i bool) {
    t := generateString(12)
    msg_escaped, _ := json.Marshal(msg)
    c.sendMessage(fmt.Sprintf(g_rtc_sendmsg_cmd, t, c.roomId, c.mUserId, id,
        msg_escaped, i, c.sessionId))
}

const g_rtc_unpublish_cmd string = string(`{"janus":"message","body":{"request":"unpublish"},` +
    `"transaction":"%s","session_id":%d,"handle_id":%d}`)

func (c *BRTCConnection) unpublish(handleId uint64) {
    t := generateString(12)
    c.sendMessage(fmt.Sprintf(g_rtc_unpublish_cmd, t, c.sessionId, handleId))
}

func (c *BRTCConnection) StopPublish() {
    c.unpublish(c.publisherHandlelId)
}

func (c *BRTCConnection) SubscribeStreaming(feedId uint64) {
    c.subscriberCreateHandle(feedId, "no")
}

const g_rtc_answer_sdp string = string(`{"janus":"message","body":{"request":"start","room":%d},` +
    `"jsep":{"type":"ANSWER","sdp":"%s"},` +
    `"transaction":"%s","session_id":%d,"handle_id":%d}`)

func (c *BRTCConnection) subscriberCreateAnswer(handleId uint64, sdp string) {
    t := generateString(12)
    c.sendMessage(fmt.Sprintf(g_rtc_answer_sdp, c.roomId, sdp, t, c.sessionId, handleId))
}

func (c *BRTCConnection) SendSubscribeAnswer(feedId uint64, sdp string) {
    h, hasH := c.FeedsHandleMap[feedId]
    if hasH {
        c.subscriberCreateAnswer(h, sdp)
    }
}

func (c *BRTCConnection) StopSubscribeStreaming(feedId uint64) {
    h, hasH := c.FeedsHandleMap[feedId]
    if hasH {
        delete(c.FeedsHandleMap, feedId)
        c.detach(h)
    }
}

func (c *BRTCConnection) doFeedLeaving(id uint64) {
    _, hasU := c.RemoteComingUsers[id]
    if hasU {
        if c.Observer != nil {
            c.Observer.OnRTCFeedLeaving(id)
        }
        delete(c.RemoteComingUsers, id)
    }
    _, hasFeed := c.FeedsHandleMap[id]
    if hasFeed {
        c.StopSubscribeStreaming(id)
    }
}

const g_rtc_detach_cmd string = string(`{"janus":"detach","transaction":"%s","session_id":%d,"handle_id":%d}`)

func (c *BRTCConnection) detach(handleId uint64) {
    t := generateString(12)
    c.sendMessage(fmt.Sprintf(g_rtc_detach_cmd, t, c.sessionId, handleId))
}

const g_rtc_destroy_cmd string = string(`{"janus":"destroy","transaction":"%s","session_id":%d}`)

func (c *BRTCConnection) destroy() {
    t := generateString(12)
    c.sendMessage(fmt.Sprintf(g_rtc_destroy_cmd, t, c.sessionId))
}

func (c *BRTCConnection) onEvent(m map[string]interface{}) {

    ihandleId, hasSender := m["sender"].(json.Number)
    if !hasSender {
        return
    }

    handleId, _ := UInt64(ihandleId)

    jh, hasHdl := c.RtcHandleMap[handleId]

    if !hasHdl {
        return
    }

    jsep, hasJsep := m["jsep"].(map[string]interface{})
    if hasJsep {
        jh.OnRemoteJsep(jh,
            replaceSdpCandidate(jsep["sdp"].(string), c.candidateIP, c.candidatePort),
            jsep["type"].(string))
    }

    pli, hasPli := m["plugindata"].(map[string]interface{})
    if hasPli {
        d, hasd := pli["data"].(map[string]interface{})
        if hasd {
            err, hasErr := d["error"].(string)
            if hasErr {
                icode, hasC := d["erro_code"].(json.Number)
                code, _ := icode.Int64()
                if hasC && c.Observer != nil {
                    c.Observer.OnRTCError(int(code), err)
                }
                return
            }

            e, hasE := d["videoroom"].(string)
            if hasE {
                if e == "joined" {
                    jh.OnJoined(jh)
                } else if e == "event" {
                    leaving, hasLeaving := d["leaving"].(json.Number)
                    if hasLeaving {
                        Name, hasName := d["display"].(string)
                        if hasName && strings.Contains(Name, "web_listener") {
                            return
                        }
                        id, _ := UInt64(leaving)
                        c.doFeedLeaving(id)
                    }
                    unpublished, hasUnpublished := d["unpublished"].(string)
                    if hasUnpublished {
                        if unpublished == "ok" && c.Observer != nil {
                            c.Observer.OnUnpublished(0, handleId)
                        }
                        return
                    }

                    feedId, hasFeed := d["unpublished"].(json.Number)
                    if hasFeed {
                        id, _ := UInt64(feedId)

                        if c.Observer != nil {
                            c.Observer.OnUnpublished(id, handleId)
                        }
                        c.doFeedLeaving(id)
                        return
                    }
                }

                pubs, hasPubs := d["publishers"].([]interface{})
                if hasPubs && (len(pubs) > 0) {
                    for _, v1 := range pubs {
                        u1, _ := v1.(map[string]interface{})
                        Id, hasId := u1["id"].(json.Number)
                        Name, hasName := u1["display"].(string)
                        Attr, hasAttr := u1["attribute"].(string)

                        var display = ""
                        if hasName {
                            display = Name
                        }

                        var attr = ""
                        if hasAttr {
                            attr = Attr
                        }

                        if hasId {
                            id, _ := UInt64(Id)
                            c.RemoteComingUsers[id] = RTCRemoteUser{display: display, attr: attr}

                            if c.Observer != nil {
                                c.Observer.OnRTCFeedComing(id, display)
                            }

                            if c.asListener && c.feedId == 0 {
                                if c.AutoSubscribe {
                                    c.subscriberCreateHandle(id, display)
                                }
                            } else if c.asListener && c.feedId != 0 && c.asPublisher {
                                if id == c.feedId {
                                    c.subscriberCreateHandle(id, display)
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

func (c *BRTCConnection) onSessionEvent(m map[string]interface{}) {
    RecvData, hasRData := m["recvdata"].(map[string]interface{})
    if hasRData {
        IsInternal, hasInter := RecvData["internal"].(bool)
        Data, hasData := RecvData["data"].(string)
        FromId, hasFrom := RecvData["from"].(json.Number)

        if !hasInter {
            IsInternal = false
        }

        if hasData && hasFrom && c.Observer != nil {
            id, _ := UInt64(FromId)
            if IsInternal {
                c.Observer.OnRTCUserAttribute(id, Data)
            } else {
                c.Observer.OnRTCUserMessage(id, Data)
            }
        }
        return
    }

    UserEvent, hasUEvent := m["userevent"].(map[string]interface{})
    if hasUEvent {
        JoinedId, hasJoined := UserEvent["joined"].(json.Number)
        Name, hasName := UserEvent["display"].(string)
        var display = ""
        if hasName {
            display = Name
        }

        if hasJoined {
            id, _ := UInt64(JoinedId)
            Attr, hasAttr := UserEvent["attribute"].(string)
            var attr = ""
            if hasAttr {
                attr = Attr
            }

            if c.Observer != nil {
                c.Observer.OnRTCUserJoinedRoom(id, display, attr)
            }
            return
        }

        LeavingId, hasLeaving := UserEvent["leaving"].(json.Number)
        if hasLeaving {
            id, _ := UInt64(LeavingId)
            if c.Observer != nil {
                c.Observer.OnRTCUserLeavingRoom(id, display)
            }
            return
        }

        Users, hasUsers := UserEvent["users"].([]interface{})
        if hasUsers {
            for _, v1 := range Users {
                u1, _ := v1.(map[string]interface{})
                Id, hasId := u1["id"].(json.Number)
                Name, hasName := u1["display"].(string)

                var display = ""
                if hasName {
                    display = Name
                }

                Attr, hasAttr := u1["attribute"].(string)
                var attr = ""
                if hasAttr {
                    attr = Attr
                }

                if hasId  && c.Observer != nil {
                    id, _ := UInt64(Id)
                    c.Observer.OnRTCUserJoinedRoom(id, display, attr)
                }
            }
        }
    }
}

func (c *BRTCConnection) onErrorEvent(m map[string]interface{}) {
    Error, hasE := m["error"].(map[string]interface{})
    if hasE {
        i, hasC := Error["code"].(json.Number)
        error_code, _ := UInt64(i)
        error_str, _ := Error["reason"].(string)
        if c.Observer != nil && hasC {
            c.Observer.OnRTCError(int(error_code), error_str)
        }
    }
}

func replaceSdpCandidate(sdp string, cip string, port uint64) (r string) {
	//a=candidate:1 1 udp 2013266431 182.61.0.35 10010 typ host\r\n";

    r = sdp
    if cip == "" && port == 0 {
        return
    }

	BRTCLog("replaceSdpCandidate, IP: %s, port: %d.", cip, port)
	s_candidate := "a=candidate"

    i_pos := strings.Index(sdp, s_candidate)
    if i_pos < 0 {
        return
    }

    t_pos := strings.Index(sdp, " typ")
    if t_pos < 0 {
        return
    }

    t := sdp[i_pos:t_pos]
    b_pos := strings.LastIndex(t, " ")
    s_port := t[b_pos:]

    nt := t[:b_pos]
    b_pos = strings.LastIndex(nt, " ")
    s_ip := nt[b_pos + 1:]

    if port != 0 {
        s_port = fmt.Sprintf("%d", port)
    }
    if cip == "" {
        cip = s_ip
    }

    s_dst := "a=candidate:1 1 udp 2013266431 " + cip + " " + s_port

    return strings.ReplaceAll(sdp, t, s_dst)
}

func (c *BRTCConnection) GetFeedIdByHandle(handleId uint64) (feedId uint64) {
    jh, hasHdl := c.RtcHandleMap[handleId]

    if hasHdl {
        return jh.feedId
    }
    return 0
} 