package baidurtc

import (
	"context"
	"errors"
	"flag"
	"fmt"
	"io"
	"log"
	"math/rand"
	"os"
	"strings"
	"testing"
	"time"

	"github.com/pion/webrtc/v4/pkg/media/h264reader"
	"github.com/pion/webrtc/v4/pkg/media/oggreader"
)

func TestMain(t *testing.T) {
	do_main()
}
func PrepareSendMedia(c *BRTCClient, l *myListener, afn string, vfn string) {
    var (
        audioFileName     = afn
        videoFileName     = vfn
        oggPageDuration   = time.Millisecond * 20
        h264FrameDuration = time.Millisecond * 33
    )
    // Assert that we have an audio or video file
    _, err := os.Stat(videoFileName)
    haveVideoFile := !os.IsNotExist(err)

    _, err = os.Stat(audioFileName)
    haveAudioFile := !os.IsNotExist(err)

    if !haveAudioFile && !haveVideoFile {
        log.Println("Could not find `" + audioFileName + "` or `" + videoFileName + "`")
        return
    }

    l.PubConnectedCtx , l.PubConnectedCtxCancel = context.WithCancel(context.Background())

    if haveVideoFile {
        go func() {
            // Open a H264 file and start reading using our IVFReader
            file, h264Err := os.Open(videoFileName)
            if h264Err != nil {
                panic(h264Err)
            }

            h264, h264Err := h264reader.NewReader(file)
            if h264Err != nil {
                panic(h264Err)
            }

            // Wait for connection established
            <- l.PubConnectedCtx.Done()

            // Send our video file frame at a time. Pace our sending so we send it at the same speed it should be played back as.
            // This isn't required since the video is timestamped, but we will such much higher loss if we send all at once.
            //
            // It is important to use a time.Ticker instead of time.Sleep because
            // * avoids accumulating skew, just calling time.Sleep didn't compensate for the time spent parsing the data
            // * works around latency issues with Sleep (see https://github.com/golang/go/issues/44343)
            ticker := time.NewTicker(h264FrameDuration)
            for ; true; <-ticker.C {
                nal, h264Err := h264.NextNAL()
                if errors.Is(h264Err, io.EOF) {
                    file.Seek(0, 0)
                    continue
                }
                if h264Err != nil {
                    continue
                }

                c.SendVideo(nal.Data, h264FrameDuration)
            }
        }()
    }

    if haveAudioFile {
        if strings.HasSuffix(afn, ".g722") || strings.HasSuffix(afn, ".pcmu") || strings.HasSuffix(afn, ".pcma"){
            go func() {
                // Open a g722 file and start reading
                file, Err := os.Open(audioFileName)
                if Err != nil {
                    return
                }

                // Wait for connection established
                <-l.PubConnectedCtx.Done()

                fi, _ := os.Stat(audioFileName)

                payload := make([]byte, fi.Size())
                _, Err = io.ReadFull(file, payload)
                if  Err != nil {
                    return
                }
                var Pos = 0;

                ticker := time.NewTicker(time.Millisecond * 20)
                for ; true; <-ticker.C {

                    c.SendAudio(payload[Pos:Pos + 160], time.Millisecond * 20)
                    Pos += 160;

                    if Pos + 160 > int(fi.Size()) {
                        Pos = 0
                    }
                }
            }()
            return
        }
        if strings.HasSuffix(afn, ".raw") || strings.HasSuffix(afn, ".raw16k") {
            go func() {
                // Open a raw file and start reading
                file, Err := os.Open(audioFileName)
                if Err != nil {
                    return
                }

                // Wait for connection established
                <-l.PubConnectedCtx.Done()

                fi, _ := os.Stat(audioFileName)

                payload := make([]byte, fi.Size())
                _, Err = io.ReadFull(file, payload)
                if  Err != nil {
                    return
                }
                var Pos = 0;

                var Datalen = 320
                if (strings.Contains(strings.ToLower(afn), "16k")) {
                    Datalen = 640
                }

                ticker := time.NewTicker(time.Millisecond * 20)
                for ; true; <-ticker.C {
                    c.SendAudioPCM(payload[Pos:Pos + Datalen], time.Millisecond * 20)
                    Pos += Datalen;
                    if Pos + Datalen > int(fi.Size()) {
                        Pos = 0
                    }
                }
            }()
            return
        }

        go func() {
            // Open a ogg file and start reading using our oggReader
            file, oggErr := os.Open(audioFileName)
            if oggErr != nil {
                panic(oggErr)
            }

            // Open on oggfile in non-checksum mode.
            ogg, _, oggErr := oggreader.NewWith(file)
            if oggErr != nil {
                panic(oggErr)
            }

            // Wait for connection established
            <-l.PubConnectedCtx.Done()

            // Keep track of last granule, the difference is the amount of samples in the buffer
            var lastGranule uint64

            // It is important to use a time.Ticker instead of time.Sleep because
            // * avoids accumulating skew, just calling time.Sleep didn't compensate for the time spent parsing the data
            // * works around latency issues with Sleep (see https://github.com/golang/go/issues/44343)
            ticker := time.NewTicker(oggPageDuration)
            for ; true; <-ticker.C {
                pageData, pageHeader, oggErr := ogg.ParseNextPage()
                if errors.Is(oggErr, io.EOF) {
                    ogg.ResetReader(func(bytesRead int64) io.Reader {
                        file.Seek(0, 0)
                        return file
                    })
                    continue
                }

                if oggErr != nil {
                    continue
                }

                // The amount of samples is the difference between the last and current timestamp
                sampleCount := float64(pageHeader.GranulePosition - lastGranule)
                lastGranule = pageHeader.GranulePosition
                sampleDuration := time.Duration((sampleCount/48000)*1000) * time.Millisecond

                c.SendAudio(pageData, sampleDuration)
            }
        }()
    }
}

type myListener struct {
    c *BRTCClient
    aw io.Writer
    vw io.Writer
    PubConnectedCtx       context.Context
    PubConnectedCtxCancel context.CancelFunc
}

func (c *myListener) OnVideoFrame(feedid uint64, video []byte, tpe string) {
    // log.Printf("OnVideoFrame feedid: %d, len: %d", feedid, len(video))
    if (c.vw != nil) {
        c.vw.Write(video)
    }
}

func (c *myListener) OnAudioData(feedid uint64, audio []byte, samlplerate int, channels int) {
    // log.Printf("OnAudioData feedid: %d, len: %d", feedid, len(audio))
    if (c.aw != nil) {
        c.aw.Write(audio)
    }
}

func (c *myListener) OnAudioPCM(feedid uint64, pcm []byte, sr int, ch int) {
    // log.Printf("OnAudioPCM feedid: %d, len: %d", feedid, len(audio))
    if (c.aw != nil) {
        c.aw.Write(pcm)
    }
}

func (c *myListener) OnTextData(feedid uint64, txt []byte) {
    log.Printf("OnTextData feedid: %d, txt: %s", feedid, string(txt))
}

func (c *myListener) OnRemoteMediaConnected(feedId uint64) {
    log.Printf("OnRemoteMediaConnected %d", feedId)
}
func (c *myListener) OnRemoteMediaDisconnected(feedId uint64) {
    log.Printf("OnRemoteMediaDisconnected %d", feedId)
}
func (c *myListener) OnLocalMediaConnected() {
    log.Printf("OnLocalMediaConnected")
    if c.PubConnectedCtxCancel != nil {
        c.PubConnectedCtxCancel()
    }
}
func (c *myListener) OnLocalMediaDisconnected() {
    log.Printf("OnLocalMediaDisconnected")
}

func do_main() {
    flag.Parse()
    log.Println("BRTC SDK Version is: " + Version())
    client := NewBRTCClient()

    con := client.B

    con.SetRoomName("Go").SetDisplayName("s").SetUserId(90000000 + rand.Uint64()%100000).SetToken("no_token")
    // con.SetPublisherSDP(string(`v=0\r\no=- 0 2 IN IP4 127.0.0.1\r\ns=-\r\nt=0 0\r\na=group:BUNDLE 0 1\r\n` +
    //     `m=audio 9 RTP\/AVP 0\r\na=rtpmap:0 PCMU\/8000\r\na=mid:0\r\na=ssrc:1718116947 cname:a\r\n` +
    //     `a=ice-ufrag:msm2\r\na=ice-pwd:Gy\r\n` +
    //     `m=video 9 RTP/AVP 102\r\na=rtpmap:102 H264\/90000\r\n` +
    //     `a=mid:1\r\na=ssrc:780489765 cname:v\r\n`))
    //
    // con.SetAutoSubscribe(false)
    // con.SetAudioCodec("pcmu")
    // .SetAsPublisher(false).SetFeedId(234)
    // con.SetMediaServerIP("110.242.70.4")
    // con.SetServerURL("ws://r.redwinner.cn:8800/brtc")
    // con.SetCandidateIP("127.0.0.1")
    // con.SetCandidatePort(8010)
    // con.SetAudioCodec("g722")

    appid_env := os.Getenv("APPID_GET_FROM_BAIDU")
    if appid_env != "" {
        con.SetAppId(appid_env)
    }

    if appid != "" {
        con.SetAppId(appid)
    }
 
    vf, _ := os.Create("dump_video.h264")
    af, _ := os.Create("dump_audio.ogg")
    my := &myListener{c: client, vw: vf, aw:af}
    client.SetMediaObserver(my)

    PrepareSendMedia(client, my,
        "../sdkdemo/res/myaudio.ogg",
        "../sdkdemo/res/myvideo.h264")

    con.Login()

    for {
        fmt.Printf("********** videocall demo V0.1 *********\n")
        fmt.Printf("Please input: h to Hangup.\n")
        fmt.Printf("              r to ReDial.\n")
        fmt.Printf("              t to LoopTest.\n")
        fmt.Printf("              q to Quit.\n")
        var commandid string

        ret, _ := fmt.Scanf("%s", &commandid)
        fmt.Printf("scanf %d.\n", ret)

        if ret < 0 {
            time.Sleep(time.Second * 200)
            continue
        } else if ret == 0 {
            time.Sleep(time.Second * 200)
            continue
        }

        switch commandid[0] {
        case 'o':
            con.Logout()
        case '1':
            con.StartPublish(string(`v=0\r\no=- 0 2 IN IP4 127.0.0.1\r\ns=-\r\nt=0 0\r\na=group:BUNDLE 0 1\r\n` +
                `m=audio 9 RTP\/AVP 0\r\na=rtpmap:0 PCMU\/8000\r\na=mid:0\r\na=ssrc:1718116947 cname:a\r\n` +
                `a=ice-ufrag:msm2\r\na=ice-pwd:Gy\r\n` +
                `m=video 9 RTP/AVP 102\r\na=rtpmap:102 H264\/90000\r\n` +
                `a=mid:1\r\na=ssrc:780489765 cname:v\r\n`))
        case '2':
            con.StopPublish()
        case '3':
            con.SubscribeStreaming(10033981)
        case '4':
            con.StopSubscribeStreaming(10033981)
        case 'm':
            con.SendMessageToUser("hello", 0)
        case 't': // loop test, t100 = 100 times.
            {

            }
        }

        if commandid[0] == 'q' {
            break
        }
    }
}

var ac string
var vc string
var af string
var adump string
var ss string
var appid string
var room string
var nv string
var na string

func init() {
    flag.StringVar(&ac, "ac", "opus", "audio codec")
    flag.StringVar(&vc, "vc", "h264", "video codec")
    flag.StringVar(&af, "af", "", "audio format")
    flag.StringVar(&adump, "adump", "", "dump audio format")
    flag.StringVar(&ss, "s", "", "server url")
    flag.StringVar(&appid, "appid", "", "AppID")
    flag.StringVar(&room, "r", "", "Room Name")
    flag.StringVar(&nv, "nv", "", "Not using video")
    flag.StringVar(&na, "na", "", "Not using audio")
}

func TestPush(t *testing.T) {
    flag.Parse()

    log.Println("BRTC SDK Version is: " + Version())
    c := NewBRTCClient()

    c.SetRoomName("Go").SetDisplayName("go TestPush").
        SetUserId(90000000 + rand.Uint64()%100000).SetToken("no_token")
    c.SetAudioCodec(ac).SetAutoSubscribe(false)

    appid_env := os.Getenv("APPID_GET_FROM_BAIDU")
    if appid_env != "" {
        c.SetAppId(appid_env)
    }

    if appid != "" {
        c.SetAppId(appid)
    }
    if room != "" {
        c.SetRoomName(room)
    }
    if ss != "" {
        c.SetServerURL(ss)
    }
    if nv != "" {
        c.SetHasVideo(false)
    }
    if na != "" {
        c.SetHasAudio(false)
    }

    ftype := ac
    if ac == "opus" {
        ftype = "ogg"
    }

    if af != "" {
        ftype = af
    }

    my := &myListener{c: c}
    c.SetMediaObserver(my)

    PrepareSendMedia(c, my,
        "../sdkdemo/res/myaudio." + ftype,
        "../sdkdemo/res/myvideo.h264")

    c.Login()

    for {
        fmt.Printf("********** videocall demo V0.1 *********\n")
        fmt.Printf("              q to Quit.\n")
        var commandid string

        ret, _ := fmt.Scanf("%s", &commandid)
        fmt.Printf("scanf %d.\n", ret)

        if ret <= 0 {
            time.Sleep(time.Second * 200)
            continue
        }

        switch commandid[0] {
        case 'o':
            c.Logout()
        case '2':
            c.StopPublish()
        }

        if commandid[0] == 'q' {
            break
        }
    }
}

func TestPull(t *testing.T) {
    flag.Parse()

    log.Println("BRTC SDK Version is: " + Version())
    c := NewBRTCClient()

    c.SetRoomName("Go").SetDisplayName("go TestPull").
        SetUserId(90000000 + rand.Uint64()%100000).SetToken("no_token")
    c.SetAudioCodec(ac).SetAutoSubscribe(true).SetAutoPublish(false)

    appid_env := os.Getenv("APPID_GET_FROM_BAIDU")
    if appid_env != "" {
        c.SetAppId(appid_env)
    }

    if appid != "" {
        c.SetAppId(appid)
    }
    if room != "" {
        c.SetRoomName(room)
    }
    if ss != "" {
        c.SetServerURL(ss)
    }

    aftype := ac
    if ac == "opus" {
        aftype = "ogg"
    }

    adtype := aftype
    if adump != "" {
        adtype = adump
        c.SetAudioOutPCM(true)
    }

    vf, _ := os.Create("dump_video." + vc)
    af, _ := os.Create("dump_audio." + adtype)
    my := &myListener{c: c, vw: vf, aw:af}
    c.SetMediaObserver(my)

    c.Login()

    for {
        fmt.Printf("********** TestPull demo V0.1 *********\n")
        fmt.Printf("              q to Quit.\n")
        var commandid string

        ret, _ := fmt.Scanf("%s", &commandid)
        fmt.Printf("scanf %d.\n", ret)

        if ret <= 0 {
            time.Sleep(time.Second * 200)
            continue
        }

        if commandid[0] == 'q' {
            break
        }
    }
}

func TestSS(t *testing.T) {
    flag.Parse()

    log.Println("BRTC SDK Version is: " + Version())
    con := NewBRTCConnection()

    con.SetRoomName("Go").SetDisplayName("TestSignal").
        SetUserId(90000000 + rand.Uint64()%100000).SetToken("no_token")

    con.SetAudioCodec(ac)

    appid_env := os.Getenv("APPID_GET_FROM_BAIDU")
    if appid_env != "" {
        con.SetAppId(appid_env)
    }

    if appid != "" {
        con.SetAppId(appid)
    }
    if room != "" {
        con.SetRoomName(room)
    }
    if ss != "" {
        con.SetServerURL(ss)
    }

    con.Login()

    for {
        fmt.Printf("********** TestSS demo V0.1 *********\n")
        fmt.Printf("              q to Quit.\n")
        var commandid string

        ret, _ := fmt.Scanf("%s", &commandid)
        fmt.Printf("scanf %d.\n", ret)

        if ret <= 0 {
            time.Sleep(time.Second * 200)
            continue
        }

        switch commandid[0] {
        case 'o':
            con.Logout()
        case '2':
            con.StopPublish()
        }

        if commandid[0] == 'q' {
            break
        }
    }
}
