// Copyright 2024 Baidu.com All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Package baidurtc implements BRTC Go SDK.
// For usage details please visite https://cloud.baidu.com/doc/RTC/index.html
package baidurtc

import (
	"errors"
	"fmt"
	"io"
	"log"
	"strings"
	"time"

	"github.com/pion/rtp"
	"github.com/pion/webrtc/v4"
	"github.com/pion/webrtc/v4/pkg/media"
	"github.com/pion/webrtc/v4/pkg/media/h264writer"
	"github.com/pion/webrtc/v4/pkg/media/h265writer"
	"github.com/pion/webrtc/v4/pkg/media/oggwriter"
)

type EventObserver interface {
    OnDisconnected()
    OnListenerNACKs(nacks uint64)
    OnNACKs(nacks uint64)

    OnRTCLoginOK()
    OnRTCLoginTimeout()
    OnRTCLoginError()
    OnRTCConnectError()
    OnRTCWebrtcUp(feedId uint64)
    OnRTCMediaStreamingEvent(feedId uint64, t int, sending bool)
    OnRTCHangUp(feedId uint64)
    OnRTCFeedComing(feedId uint64, name string)
    OnRTCFeedLeaving(feedId uint64)
    OnRTCUserMessage(feedId uint64, msg string)
    OnRTCUserAttribute(feedId uint64, attr string)
    OnRTCUserJoinedRoom(feedId uint64, name string, attr string)
    OnRTCUserLeavingRoom(feedId uint64, name string)
    OnRTCError(error_code int, e string)

    OnPublisherJoined()
    OnRemoteMediaUnpublished(feedId uint64)

    OnLocalMediaPublishedOK()
    OnLocalMediaUnpublished()
}

const (
    // Input and output formats
    MediaTypeH264 = iota
    MediaTypeH263
    MediaTypeH265
    MediaTypeH266
    MediaTypeAV1
    MediaTypeJPEG
    MediaTypeOPUS
    MediaTypeG722
    MediaTypePCMU
    MediaTypePCMA
    MediaTypeAAC
    MediaTypeAMRNB
    MediaTypeAMRWB
)

type MediaObserver interface {
    OnVideoFrame(feedid uint64, video []byte, tpe string)
    OnAudioData(feedid uint64, audio []byte, samlplerate int, channels int)
    OnAudioPCM(feedid uint64, pcm []byte, sr int, ch int)
    OnTextData(feedid uint64, txt []byte)

    OnLocalMediaConnected()
    OnLocalMediaDisconnected()
    OnRemoteMediaConnected(feedId uint64)
    OnRemoteMediaDisconnected(feedId uint64)
}

type MediaWriter struct {
    mo       MediaObserver
    feedId   uint64
    isVideo  bool
    usingPCM bool
    mType    int
    g722_dec *G722Decoder
    opus_dec *OPUSDecoder
}

func (w *MediaWriter) Write(p []byte) (n int, err error) {
    if w.mo != nil && len(p) > 0 {
        if w.isVideo {
            w.mo.OnVideoFrame(w.feedId, p, "h264")
        } else if w.usingPCM {
            switch w.mType {
                case MediaTypePCMU:
                    p = DecodeUlaw(p)
                case MediaTypePCMA:
                    p = DecodeAlaw(p)
                case MediaTypeG722:

                    if w.g722_dec == nil {
                        w.g722_dec = NewG722Decoder(RateDefault, 0)
                    }
                    if w.g722_dec != nil {
                        n := len(p) * 2
                        out := make([]int16, n)
                        w.g722_dec.Decode(out, p)

                        bytes := make([]byte, n * 2)
                        for i := 0; i < n; i++ {
                            bytes[i*2] = byte(out[i])
                            bytes[i*2+1] = byte(out[i] >> 8)
                        }
                        p = bytes
                    }
                case MediaTypeOPUS:
                    if w.opus_dec == nil {
                        w.opus_dec = NewOPUSDecoder(48000)
                    }
                    if w.opus_dec != nil {
                        w.opus_dec.Decode(p, p)
                    }
                default:
            }

            w.mo.OnAudioPCM(w.feedId, p, 48000, 2)
        } else {
            w.mo.OnAudioData(w.feedId, p, 48000, 2)
        }

        return len(p), nil
    }
    return 0, nil
}

type (
    AudPacketWriter struct {
        writer       io.Writer
    }
)

// NewWith initializes a new writer with an io.Writer output
func AudPacketWriter_NewWith(w io.Writer) *AudPacketWriter {
    return &AudPacketWriter{
        writer: w,
    }
}

// WriteRTP adds a new packet and writes the appropriate headers for it
func (h *AudPacketWriter) WriteRTP(packet *rtp.Packet) error {
    if len(packet.Payload) == 0 {
        return nil
    }
    _, err := h.writer.Write(packet.Payload)
    return err
}

// Close closes the underlying writer
func (h *AudPacketWriter) Close() error {
    if h.writer != nil {
        if closer, ok := h.writer.(io.Closer); ok {
            return closer.Close()
        }
    }
    return nil
}

type BRTCClient struct {
    B                     *BRTCConnection
    pcs                   map[uint64]*webrtc.PeerConnection
    publisherHandlelId    uint64
    videoTrack            *webrtc.TrackLocalStaticSample
    audioTrack            *webrtc.TrackLocalStaticSample
    dataChannel           *webrtc.DataChannel
    mo                    MediaObserver
    eo                    EventObserver
    g722_enc              *G722Encoder
    opus_enc              *OPUSEncoder
    audioOutPCM           bool
    mAudioType            int
    audioOutOpusInOgg     bool
}

func (c *BRTCClient) OnDisconnected() {
    BRTCLog("onDisconnected \n")
    if c.eo != nil {
        c.eo.OnDisconnected()
    }
}

func (c *BRTCClient) OnPublisherJoined(handleId uint64) {
    BRTCLog("onPublisherJoined handle: %d\n", handleId)
    if c.B.AutoPublish {
        c.newSendPCforHandle(handleId)
    }

    c.publisherHandlelId = handleId
    if c.eo != nil {
        c.eo.OnPublisherJoined()
    }
}

func (c *BRTCClient) OnPublisherRemoteJsep(handleId uint64, sdp string, typ string) {
    BRTCLog("onPublisherRemoteJsep handle: %d, typ:%s, sdp: %s\n", handleId, typ, sdp)
    pc, hasPC := c.pcs[handleId]

    if hasPC {
        jsep := webrtc.SessionDescription{}
        jsep.SDP = sdp
        jsep.Type = webrtc.SDPTypeAnswer

        pc.SetRemoteDescription(jsep)
    }
}

func (c *BRTCClient) AddTrackForSending(pc *webrtc.PeerConnection) {
    if c.B.HasVideo {
        videoMime := webrtc.MimeTypeH264
        vc := strings.ToUpper(c.B.VideoCodec)
        switch vc {
        case "H264":
            videoMime = webrtc.MimeTypeH264
        default:
            videoMime = "video/" + vc
        }
        // Create a video track
        videoTrack, videoTrackErr := webrtc.NewTrackLocalStaticSample(webrtc.RTPCodecCapability{MimeType: videoMime}, "video", "pion")
        if videoTrackErr != nil {
            panic(videoTrackErr)
        }

        rtpSender, videoTrackErr := pc.AddTrack(videoTrack)
        if videoTrackErr != nil {
            panic(videoTrackErr)
        }

        c.videoTrack = videoTrack
        // Read incoming RTCP packets
        // Before these packets are returned they are processed by interceptors. For things
        // like NACK this needs to be called.
        go func() {
            rtcpBuf := make([]byte, 1500)
            for {
                if _, _, rtcpErr := rtpSender.Read(rtcpBuf); rtcpErr != nil {
                    return
                }
            }
        }()
    }

    if c.B.HasAudio {
        audioMime := webrtc.MimeTypeOpus
        ac := strings.ToUpper(c.B.AudioCodec)
        audioMime = "audio/" + ac
        switch ac {
        case "OPUS":
            audioMime = webrtc.MimeTypeOpus
            c.mAudioType = MediaTypeOPUS
        case "PCMU":
            c.mAudioType = MediaTypePCMU
        case "PCMA":
            c.mAudioType = MediaTypePCMA
        case "G722":
            c.mAudioType = MediaTypeG722
        default:
            c.mAudioType = MediaTypeOPUS
        }

        // Create a audio track
        audioTrack, audioTrackErr := webrtc.NewTrackLocalStaticSample(webrtc.RTPCodecCapability{MimeType: audioMime}, "audio", "pion")
        if audioTrackErr != nil {
            panic(audioTrackErr)
        }

        rtpSender, audioTrackErr := pc.AddTrack(audioTrack)
        if audioTrackErr != nil {
            panic(audioTrackErr)
        }

        c.audioTrack = audioTrack
        // Read incoming RTCP packets
        // Before these packets are returned they are processed by interceptors. For things
        // like NACK this needs to be called.
        go func() {
            rtcpBuf := make([]byte, 1500)
            for {
                if _, _, rtcpErr := rtpSender.Read(rtcpBuf); rtcpErr != nil {
                    return
                }
            }
        }()
    }
}

func getUfragFromSDP(sdp string) (r string, e error) {
    i_pos := strings.Index(sdp, "a=ice-ufrag:")
    if i_pos < 0 {
        return "", errors.New("no ufrag")
    }

    ufrag := sdp[i_pos:]
    t_pos := strings.Index(ufrag, "\r\n")
    if t_pos < 0 {
        return "", errors.New("no ufrag")
    }

    t := ufrag[12:t_pos]

    return t, nil
}

func (c *BRTCClient) StartPublish() {
    if c.publisherHandlelId != 0 {
        c.newSendPCforHandle(c.publisherHandlelId)
    }
}

func (c *BRTCClient) StopPublish() {
    c.B.StopPublish()
    c.publisherHandlelId = 0
}

func (c *BRTCClient) SubscribeStreaming(feedId uint64) {
    c.B.SubscribeStreaming(feedId)
}

func (c *BRTCClient) StopSubscribeStreaming(feedId uint64) {
    c.B.StopSubscribeStreaming(feedId)
}

func (c *BRTCClient) newSendPCforHandle(handleId uint64) {
    pc, hasPC := c.pcs[handleId]

    if hasPC {
        pc.Close()
    }

    // Prepare the configuration
    config := webrtc.Configuration{}

    // Create a new RTCPeerConnection
    peerConnection, err := webrtc.NewPeerConnection(config)
    if err != nil {
        panic(err)
    }

    c.AddTrackForSending(peerConnection)

    if c.B.HasData {
        // Create a datachannel with label 'data'
        c.dataChannel, _ = peerConnection.CreateDataChannel("JanusDataChannel", nil)
    }

    // Create an answer to send to the other process
    offer, err := peerConnection.CreateOffer(nil)
    if err != nil {
        BRTCLog("peerConnection.CreateAnswer err: %s\n", err)
    }

    c.B.StartPublish(strings.Replace(offer.SDP, "\r\n", `\r\n`, -1))

    // Set the handler for Peer connection state
    // This will notify you when the peer has connected/disconnected
    peerConnection.OnConnectionStateChange(func(s webrtc.PeerConnectionState) {
        BRTCLog("Peer Connection State has changed: %s\n", s.String())

        if s == webrtc.PeerConnectionStateFailed {
            log.Println("Peer Connection has gone to failed exiting")
            if c.mo != nil {
                c.mo.OnLocalMediaDisconnected()
            }
        }

        if s == webrtc.PeerConnectionStateClosed {
            // PeerConnection was explicitly closed. This usually happens from a DTLS CloseNotify
            log.Println("Peer Connection has gone to closed exiting")
            if c.mo != nil {
                c.mo.OnLocalMediaDisconnected()
            }
        }

        if s == webrtc.PeerConnectionStateConnected {

            if c.mo != nil {
                c.mo.OnLocalMediaConnected()
            }
        }

        if s == webrtc.PeerConnectionStateDisconnected {
            if c.mo != nil {
                c.mo.OnLocalMediaDisconnected()
            }
        }
    })

    // Register data channel creation handling
    peerConnection.OnDataChannel(func(d *webrtc.DataChannel) {
        BRTCLog("New DataChannel %s %d\n", d.Label(), d.ID())

        // Register channel opening handling
        d.OnOpen(func() {
            BRTCLog("Data channel '%s'-'%d' open.\n", d.Label(), d.ID())
        })

        // Register text message handling
        d.OnMessage(func(msg webrtc.DataChannelMessage) {
            BRTCLog("Message from DataChannel '%s': '%s'\n", d.Label(), string(msg.Data))
        })
    })

    ufrag, e := getUfragFromSDP(offer.SDP)
    if (e == nil) {
        BRTCLog("getUfragFromSDP: %s\n", ufrag)
    }

    peerConnection.OnICECandidate(func(i *webrtc.ICECandidate) {
        if i == nil {
            c.B.TrickleCandicateComplete(handleId)
            return
        }
        c.B.TrickleCandicate(handleId,
            `candidate:342623 1 udp 2122260223 127.0.0.1 65470 typ host generation 0 ufrag ` + ufrag + ` network-id 1 network-cost 10`)
    })

    // Sets the LocalDescription, and starts our UDP listeners
    err = peerConnection.SetLocalDescription(offer)
    if err != nil {
        BRTCLog("peerConnection.SetLocalDescription err: %s\n", err)
    }

    c.pcs[handleId] = peerConnection
}

func saveToWriter(i media.Writer, track *webrtc.TrackRemote, mo MediaObserver, feedId uint64) {
    defer func() {
        if err := i.Close(); err != nil {
            panic(err)
        }
    }()

    for {
        rtpPacket, _, err := track.ReadRTP()
        if err != nil {
            fmt.Println(err)
            return
        }

        if err := i.WriteRTP(rtpPacket); err != nil {
            fmt.Println(err)
            return
        }
    }
}

func (c *BRTCClient) newRecievePCforHandle(handleId uint64, feedId uint64, sdp string, typ string) {
    pc, hasPC := c.pcs[handleId]

    if hasPC {
        pc.Close()
    }

    // Prepare the configuration
    config := webrtc.Configuration{}

    // Create a new RTCPeerConnection

    answerSettingEngine := webrtc.SettingEngine{}
    answerSettingEngine.SetLite(false)
    peerConnection, err := webrtc.NewAPI(webrtc.WithSettingEngine(answerSettingEngine)).NewPeerConnection(config)
    // peerConnection, err := webrtc.NewPeerConnection(config)
    if err != nil {
        panic(err)
    }

    jsep := webrtc.SessionDescription{}
    jsep.SDP = sdp
    jsep.Type = webrtc.SDPTypeOffer

    if err := peerConnection.SetRemoteDescription(jsep); err != nil {
        BRTCLog("peerConnection.SetRemoteDescription err: %s\n", err)
    }

    // Create an answer to send to the other process
    answer, err := peerConnection.CreateAnswer(nil)
    if err != nil {
        BRTCLog("peerConnection.CreateAnswer err: %s\n", err)
    }

    c.B.SendSubscribeAnswer(feedId, strings.Replace(answer.SDP, "\r\n", `\r\n`, -1))

    // Set the handler for Peer connection state
    // This will notify you when the peer has connected/disconnected
    peerConnection.OnConnectionStateChange(func(s webrtc.PeerConnectionState) {
        BRTCLog("Peer Connection State has changed: %s\n", s.String())

        if s == webrtc.PeerConnectionStateFailed {
            if c.mo != nil {
                c.mo.OnRemoteMediaDisconnected(feedId)
            }
            log.Println("Peer Connection has gone to failed exiting")
        }

        if s == webrtc.PeerConnectionStateClosed {
            // PeerConnection was explicitly closed. This usually happens from a DTLS CloseNotify
            log.Println("Peer Connection has gone to closed exiting")
            if c.mo != nil {
                c.mo.OnRemoteMediaDisconnected(feedId)
            }
        }

        if s == webrtc.PeerConnectionStateConnected {
            if c.mo != nil {
                c.mo.OnRemoteMediaConnected(feedId)
            }
        }
    })

    // Register data channel creation handling
    peerConnection.OnDataChannel(func(d *webrtc.DataChannel) {
        BRTCLog("New DataChannel %s %d\n", d.Label(), d.ID())

        // Register channel opening handling
        d.OnOpen(func() {
            BRTCLog("Data channel '%s'-'%d' open.\n", d.Label(), d.ID())
        })

        // Register text message handling
        d.OnMessage(func(msg webrtc.DataChannelMessage) {
            BRTCLog("Message from DataChannel '%s': '%s'\n", d.Label(), string(msg.Data))
            if c.mo != nil {
                c.mo.OnTextData(feedId, msg.Data)
            }
        })
    })

    ufrag, e := getUfragFromSDP(answer.SDP)
    if (e != nil) {
        BRTCLog("getUfragFromSDP: %s\n", ufrag)
    }

    peerConnection.OnICECandidate(func(i *webrtc.ICECandidate) {
        if i == nil {
            c.B.TrickleCandicateComplete(handleId)
            return
        }
        c.B.TrickleCandicate(handleId,
            `candidate:342623 1 udp 2122260223 127.0.0.1 65470 typ host generation 0 ufrag ` + ufrag + ` network-id 1 network-cost 10`)
    })

    // Sets the LocalDescription, and starts our UDP listeners
    err = peerConnection.SetLocalDescription(answer)
    if err != nil {
        BRTCLog("peerConnection.SetLocalDescription err: %s\n", err)
    }

    // Allow us to receive 1 audio track, and 1 video track
    if _, err = peerConnection.AddTransceiverFromKind(webrtc.RTPCodecTypeAudio); err != nil {
        panic(err)
    } else if _, err = peerConnection.AddTransceiverFromKind(webrtc.RTPCodecTypeVideo); err != nil {
        panic(err)
    }

    w := &MediaWriter{mo: c.mo, feedId: feedId, isVideo: false, usingPCM: c.audioOutPCM}
    w2 := &MediaWriter{mo: c.mo, feedId: feedId, isVideo: true}

    // Set a handler for when a new remote track starts, this handler saves buffers to disk as
    // an ivf file, since we could have multiple video tracks we provide a counter.
    // In your application this is where you would handle/process video
    peerConnection.OnTrack(func(track *webrtc.TrackRemote, receiver *webrtc.RTPReceiver) { //nolint: revive
        codec := track.Codec()
        if strings.EqualFold(codec.MimeType, webrtc.MimeTypeOpus) {
            BRTCLog("Got Opus track, saving to disk as output.ogg (48 kHz, 2 channels)")
            w.mType = MediaTypeOPUS
            if !c.audioOutOpusInOgg {
                audFile := AudPacketWriter_NewWith(w)
                saveToWriter(audFile, track, c.mo, feedId)
                return
            }
            oggFile, _ := oggwriter.NewWith(w, 48000, 2)
            saveToWriter(oggFile, track, c.mo, feedId)
        } else if strings.EqualFold(codec.MimeType, webrtc.MimeTypeH264) {
            BRTCLog("Got H264 track, saving to disk as output.h264")
            w2.mType = MediaTypeH264
            h264File := h264writer.NewWith(w2)
            saveToWriter(h264File, track, c.mo, feedId)
        }  else if strings.EqualFold(codec.MimeType, webrtc.MimeTypeH265) {
            BRTCLog("Got H265 track, saving to disk as output.h265")
            w2.mType = MediaTypeH265
            h265File := h265writer.NewWith(w2)
            saveToWriter(h265File, track, c.mo, feedId)
        } else if strings.EqualFold(codec.MimeType, webrtc.MimeTypeG722) || 
                    strings.EqualFold(codec.MimeType, webrtc.MimeTypePCMU) ||
                    strings.EqualFold(codec.MimeType, webrtc.MimeTypePCMA) {
            BRTCLog("Got %s track, saving to disk as dump file", codec.MimeType)

            switch codec.MimeType {
                case "audio/G722":
                    w.mType = MediaTypeG722
                case "audio/PCMU":
                    w.mType = MediaTypePCMU
                case "audio/PCMA":
                    w.mType = MediaTypePCMA
                default:
                    w.mType = MediaTypePCMU
            }

            audFile := AudPacketWriter_NewWith(w)
            saveToWriter(audFile, track, c.mo, feedId)
        }
    })

    c.pcs[handleId] = peerConnection
}

func (c *BRTCClient) OnSubscriberHandleRemoteJsep(handleId uint64,
    feedId uint64, sdp string, typ string) {
    BRTCLog("onSubscriberHandleRemoteJsep handle: %d, feedId: %d, typ: %s, sdp: %s\n", handleId, feedId, typ, sdp)
    c.newRecievePCforHandle(handleId, feedId, sdp, typ)
}

func (c *BRTCClient) OnRTCFeedLeaving(feedId uint64) {
    BRTCLog("onRTCFeedLeaving id:%d\n", feedId)
    if c.eo != nil {
        c.eo.OnRTCFeedLeaving(feedId)
    }
}

func (c *BRTCClient) OnListenerNACKs(nacks uint64) {
    BRTCLog("onListenerNACKs nacks: %d\n", nacks)
}

func (c *BRTCClient) OnNACKs(nacks uint64) {
    BRTCLog("onNACKs nacks: %d\n", nacks)
    if c.eo != nil {
        c.eo.OnNACKs(nacks)
    }
}

func (c *BRTCClient) OnUnpublished(feedId uint64, handleId uint64) {
    BRTCLog("onUnpublished id:%d, handle: %d\n", feedId, handleId)

    pc, hasPC := c.pcs[handleId]
    if hasPC {
        pc.Close()
        delete(c.pcs, handleId)
    }
}

func (c *BRTCClient) OnRTCLoginOK() {
    BRTCLog("onRTCLoginOK\n")
    if c.eo != nil {
        c.eo.OnRTCLoginOK()
    }
}

func (c *BRTCClient) OnRTCLoginTimeout() {
    BRTCLog("onRTCLoginTimeout\n")
    if c.eo != nil {
        c.eo.OnRTCLoginTimeout()
    }
}

func (c *BRTCClient) OnRTCLoginError() {
    BRTCLog("onRTCLoginError\n")
    if c.eo != nil {
        c.eo.OnRTCLoginError()
    }
}

func (c *BRTCClient) OnRTCConnectError() {
    BRTCLog("onRTCConnectError\n")
    if c.eo != nil {
        c.eo.OnRTCLoginError()
    }
}

func (c *BRTCClient) OnRTCWebrtcUp(handleId uint64) {
    if c.eo != nil {
        c.eo.OnRTCWebrtcUp(c.B.GetFeedIdByHandle(handleId))
    }
}

func (c *BRTCClient) OnRTCMediaStreamingEvent(handleId uint64, t int, sending bool) {
    if c.eo != nil {
        c.eo.OnRTCMediaStreamingEvent(c.B.GetFeedIdByHandle(handleId), t, sending)
    }
}

func (c *BRTCClient) OnRTCHangUp(handleId uint64) {
    if c.eo != nil {
        c.eo.OnRTCHangUp(c.B.GetFeedIdByHandle(handleId))
    }
}

func (c *BRTCClient) OnRTCFeedComing(feedId uint64, name string) {
    BRTCLog("onRTCFeedComing id: %d, display: %s\n", feedId, name)
    if c.eo != nil {
        c.eo.OnRTCFeedComing(feedId, name)
    }
}

func (c *BRTCClient) OnRTCUserMessage(feedId uint64, msg string) {
    BRTCLog("onRTCUserMessage id: %d, msg: %s\n", feedId, msg)
    if c.eo != nil {
        c.eo.OnRTCUserMessage(feedId, msg)
    }
}

func (c *BRTCClient) OnRTCUserAttribute(feedId uint64, attr string) {
    BRTCLog("onRTCUserAttribute id: %d, msg: %s\n", feedId, attr)
    if c.eo != nil {
        c.eo.OnRTCUserAttribute(feedId, attr)
    }
}

func (c *BRTCClient) OnRTCUserJoinedRoom(feedId uint64, name string, attr string) {
    BRTCLog("onRTCUserJoinedRoom id: %d, display: %s, attr: %s\n", feedId, name, attr)
    if c.eo != nil {
        c.eo.OnRTCUserJoinedRoom(feedId, name, attr)
    }
}

func (c *BRTCClient) OnRTCUserLeavingRoom(feedId uint64, name string) {
    BRTCLog("onRTCUserLeavingRoom id: %d, display: %s\n", feedId, name)
    if c.eo != nil {
        c.eo.OnRTCUserLeavingRoom(feedId, name)
    }
}

func (c *BRTCClient) OnRTCError(error_code int, e string) {
    BRTCLog("onRTCError error: %d, reson: %s\n", error_code, e)
    if c.eo != nil {
        c.eo.OnRTCError(error_code, e)
    }
}

func NewBRTCClient() (c *BRTCClient) {
    c = &BRTCClient{
        B:           NewBRTCConnection(),
        pcs:         make(map[uint64]*webrtc.PeerConnection, 0),
        audioOutPCM: false,
        audioOutOpusInOgg: false,
    }

    c.B.setSDKTag("BRTC.Go.SDK " + sdkVersion)
    c.B.SetObserver(c)
    return
}

func (c *BRTCClient) SendVideo(f []byte, d time.Duration) error {
    if c.videoTrack != nil {
        if h264Err := c.videoTrack.WriteSample(media.Sample{Data: f, Duration: d}); h264Err != nil {
            return h264Err
        }
    }
    return nil
}

func (c *BRTCClient) SendAudio(f []byte, d time.Duration) error {
    if c.audioTrack != nil {
        if oggErr := c.audioTrack.WriteSample(media.Sample{Data: f, Duration: d}); oggErr != nil {
            return oggErr
        }
    }
    return nil
}

func (c *BRTCClient) SendAudioPCM(f []byte, d time.Duration) error {
    if c.audioTrack != nil {

        e := f
        switch c.mAudioType {
        case MediaTypeG722:
            l := (len(f) + 1 ) / 2

            if (l < 160 || (l % 2 != 0)) {
                return nil
            }

            f_int16 := make([]int16, l)
            for i := 0; i + 1 < len(f); i += 2 {
                f_int16[i/2] = int16(uint16(f[i]) | (uint16(f[i+1]) << 8))
            }

            // fmt.Printf("int16 len=%d %#v", len(f_int16), f_int16)
            if c.g722_enc == nil {
                c.g722_enc = NewG722Encoder(RateDefault, 0)
            }

            if c.g722_enc != nil {
                e = make([]byte, (l + 1) / 2)
                c.g722_enc.Encode(e, f_int16)
            }
            // fmt.Printf("encoded len=%d %#v", len(e), e)
            // c.mo.OnAudioData(0, e, 16000, 1)
        case MediaTypePCMU:
            e = EncodeUlaw(f)
        case MediaTypePCMA:
            e = EncodeAlaw(f)
        case MediaTypeOPUS:
            if c.opus_enc == nil {
                c.opus_enc = NewOPUSEncoder(48000, 2)
            }

            if c.opus_enc != nil {
                e = make([]byte, len(f))
                c.opus_enc.Encode(e, f)
            }
        default:
            e = f
        }

        if Err := c.audioTrack.WriteSample(media.Sample{Data: e, Duration: d}); Err != nil {
            return Err
        }
    }
    return nil
}

func (c *BRTCClient) SendData(t string) error {
    if c.dataChannel != nil {
        if dErr := c.dataChannel.SendText(t); dErr != nil {
            return dErr
        }
    }
    return nil
}

func (c *BRTCClient) SendRawData(d []byte) error {
    if c.dataChannel != nil {
        if dErr := c.dataChannel.Send(d); dErr != nil {
            return dErr
        }
    }
    return nil
}

func (c *BRTCClient) SetMediaObserver(mo MediaObserver) *BRTCClient {
    c.mo = mo
    return c
}

func (c *BRTCClient) SetEventObserver(eo EventObserver) *BRTCClient {
    c.eo = eo
    return c
}

func (c *BRTCClient) SetAudioOutPCM(pcm bool) *BRTCClient {
    c.audioOutPCM = pcm
    return c
}

func (c *BRTCClient) SetAudioOutOgg(ogg bool) *BRTCClient {
    c.audioOutOpusInOgg = ogg
    return c
}

func (c *BRTCClient) Login() {
    c.B.Login()
}

func (c *BRTCClient) Logout() {
    c.B.Logout()
}

func (c *BRTCClient) SetAppId(a string) (ret *BRTCClient) {
    c.B.SetAppId(a)
    return c
}

func (c *BRTCClient) SetRoomName(r string) (ret *BRTCClient) {
    c.B.SetRoomName(r)
    return c
}

func (c *BRTCClient) SetUserId(u uint64) (ret *BRTCClient) {
    c.B.SetUserId(u)
    return c
}

func (c *BRTCClient) SetDisplayName(d string) (ret *BRTCClient) {
    c.B.SetDisplayName(d)
    return c
}

func (c *BRTCClient) SetToken(t string) (ret *BRTCClient) {
    c.B.SetToken(t)
    return c
}

func (c *BRTCClient) SetAudioCodec(ac string) (ret *BRTCClient) {
    c.B.SetAudioCodec(ac)
    return c
}

func (c *BRTCClient) SetVideoCodec(vc string) (ret *BRTCClient) {
    c.B.SetVideoCodec(vc)
    return c
}

func (c *BRTCClient) SetAsPublisher(b bool) (ret *BRTCClient) {
    c.B.SetAsPublisher(b)
    return c
}

func (c *BRTCClient) SetAsListener(b bool) (ret *BRTCClient) {
    c.B.SetAsListener(b)
    return c
}

func (c *BRTCClient) SetHasAudio(b bool) (ret *BRTCClient) {
    c.B.SetHasAudio(b)
    return c
}

func (c *BRTCClient) SetHasVideo(b bool) (ret *BRTCClient) {
    c.B.SetHasVideo(b)
    return c
}

func (c *BRTCClient) SetHasData(b bool) (ret *BRTCClient) {
    c.B.HasData = b
    return c
}

func (c *BRTCClient) SetFeedId(i uint64) (ret *BRTCClient) {
    c.B.SetFeedId(i)
    return c
}

func (c *BRTCClient) SetAutoPublish(b bool) (ret *BRTCClient) {
    c.B.SetAutoPublish(b)
    return c
}

func (c *BRTCClient) SetAutoSubscribe(b bool) (ret *BRTCClient) {
    c.B.SetAutoSubscribe(b)
    return c
}

func (c *BRTCClient) SetMediaServerIP(ip string) (ret *BRTCClient) {
    c.B.SetMediaServerIP(ip)
    return c
}

func (c *BRTCClient) SetServerURL(s string) (ret *BRTCClient) {
    c.B.SetServerURL(s)
    return c
}

func (c *BRTCClient) SetCandidateIP(ip string) (ret *BRTCClient) {
    c.B.SetCandidateIP(ip)
    return c
}

func (c *BRTCClient) SetCandidatePort(p uint64) (ret *BRTCClient) {
    c.B.SetCandidatePort(p)
    return c
}

func (c *BRTCClient) SendMessageToUser(msg string, id uint64) {
    c.B.SendMessageToUser(msg, id)
}