/*****************************************************************************
* y4m.c: YUV4MPEG stream parser.
*****************************************************************************
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111, USA.
*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <inttypes.h>
#include <malloc.h>
#include <string.h>
#include "y4m.h"

/* Most of this is from x264 */



#define Y4M_MAGIC "YUV4MPEG2"
#define MAX_YUV4_HEADER 80
#define Y4M_FRAME_MAGIC "FRAME"
#define MAX_FRAME_HEADER 80

int open_file_y4m(char *filename, y4m_input_t **handle)
{
    int i, n, d;
    int interlaced;
    char header[MAX_YUV4_HEADER + 10];
    char *tokstart, *tokend, *header_end;
    y4m_input_t *h = (y4m_input_t *)calloc(1, sizeof(*h));

    h->next_frame = 0;

    if (!strcmp(filename, "-"))
        h->fp = stdin;
    else
        h->fp = fopen(filename, "rb");
    if (h->fp == NULL)
    {
        fprintf(stderr, "Can not open y4m file, please check file path.\n");
        free(h);
        return -1;
    }


    h->frame_header_len = strlen(Y4M_FRAME_MAGIC) + 1;

    /* Read header */
    for (i = 0; i < MAX_YUV4_HEADER; i++) {
        header[i] = fgetc((FILE *)h->fp);
        if (header[i] == '\n') {
            /* Add a space after last option. Makes parsing "444" vs
               "444alpha" easier. */
            header[i + 1] = 0x20;
            header[i + 2] = 0;
            break;
        }
    }
    if (i == MAX_YUV4_HEADER || strncmp(header, Y4M_MAGIC, strlen(Y4M_MAGIC)))
    {
        fprintf(stderr, "not y4m file\n");
        free(h);
        return -1;
    }

    /* Scan properties */
    header_end = &header[i + 1];        /* Include space */
    h->seq_header_len = i + 1;
    for (tokstart = &header[strlen(Y4M_MAGIC) + 1]; tokstart < header_end; tokstart++) {
        if (*tokstart == 0x20)
            continue;
        switch (*tokstart++) {
            case 'W':              /* Width. Required. */
                h->width = strtol(tokstart, &tokend, 10);
                tokstart = tokend;
                break;
            case 'H':              /* Height. Required. */
                h->height = strtol(tokstart, &tokend, 10);
                tokstart = tokend;
                break;
            case 'C':              /* Color space */
                if (strncmp("420", tokstart, 3)) {
                    fprintf(stderr, "Colorspace unhandled\n");
                    free(h);
                    return -1;
                }
                tokstart = strchr(tokstart, 0x20);
                break;
            case 'I':              /* Interlace type */
                switch (*tokstart++) {
                    case 'p':
                        interlaced = 0;
                        break;
                    case '?':
                    case 't':
                    case 'b':
                    case 'm':
                    default:
                        interlaced = 1;
                        fprintf(stderr, "Warning, this sequence might be interlaced\n");
                }
                break;
            case 'F': /* Frame rate - 0:0 if unknown */
                      /* Frame rate in unimportant. */
                if (sscanf(tokstart, "%d:%d", &n, &d) == 2 && n && d) {
                    h->fps_num = n;
                    h->fps_den = d;
                }
                tokstart = strchr(tokstart, 0x20);
                break;
            case 'A': /* Pixel aspect - 0:0 if unknown */
                      /* Don't override the aspect ratio if sar has been explicitly set on the commandline. */
                if (sscanf(tokstart, "%d:%d", &n, &d) == 2 && n && d) {
                    h->par_width = n;
                    h->par_height = d;
                }
                tokstart = strchr(tokstart, 0x20);
                break;
            case 'X': /* Vendor extensions */
                if (!strncmp("YSCSS=", tokstart, 6)) {
                    /* Older nonstandard pixel format representation */
                    tokstart += 6;
                    if (strncmp("420JPEG", tokstart, 7) &&
                         strncmp("420MPEG2", tokstart, 8) &&
                         strncmp("420PALDV", tokstart, 8)) {
                        fprintf(stderr, "Unsupported extended colorspace\n");
                        free(h);
                        return -1;
                    }
                }
                tokstart = strchr(tokstart, 0x20);
                break;
        }
    }

    fprintf(stderr, "yuv4mpeg: %ix%i@%i/%ifps, %i:%i\n",
            h->width, h->height, h->fps_num, h->fps_den,
            h->par_width, h->par_height);

    *handle = h;
    return 0;
}

int read_frame_y4m(y4m_input_t *handle, int framenum, unsigned char *buf, int len)
{
    int slen = strlen(Y4M_FRAME_MAGIC);
    int i = 0;
    char header[16];
    y4m_input_t *h = handle;

    if (framenum != h->next_frame) {
        if (fseek((FILE *)h->fp, (uint64_t)framenum*(3*(h->width*h->height)/2+h->frame_header_len)
             + h->seq_header_len, SEEK_SET))
            return -1;
    }

    /* Read frame header - without terminating '\n' */
    if (fread(header, 1, slen, (FILE *)h->fp) != slen)
        return -1;

    header[slen] = 0;
    if (strncmp(header, Y4M_FRAME_MAGIC, slen)) {
        fprintf(stderr, "Bad header magic (%x <=> %s)\n",
                *((uint32_t *) header), header);
        return -1;
    }

    /* Skip most of it */
    while (i < MAX_FRAME_HEADER && fgetc((FILE *)h->fp) != '\n')
        i++;
    if (i == MAX_FRAME_HEADER) {
        fprintf(stderr, "Bad frame header!\n");
        return -1;
    }
    h->frame_header_len = i + slen + 1;

    if (fread(buf,1,h->width*h->height *3 /2,(FILE *)h->fp) <=0 )
        return -1;

    h->next_frame = framenum + 1;

    return 0;
}

int close_file_y4m(y4m_input_t *handle)
{
    y4m_input_t *h = handle;
    if (!h || !h->fp)
        return 0;
    fclose((FILE *)h->fp);
    free(h);
    return 0;
}
