// Copyright 2020 baidu.com. All rights reserved.

#include "UE4Connection.h"
#include "thread"

using PixelStreamingProtocol::EToUE4Msg;
using PixelStreamingProtocol::EToProxyMsg;

#ifndef __EXT_POSIX1_198808
#define __EXT_POSIX1_198808
#endif

#include <iostream> //cout
#include <stdio.h> //printf
#include <stdlib.h>
#include <string.h> //strlen
#include <string> //string
#include <sys/socket.h> //socket
#include <arpa/inet.h> //inet_addr
#include <netdb.h> //hostent
#include <fcntl.h>

/**
TCP Client class
*/
class tcp_client
{
    private:
        int sock;
        std::string address;
        int port;
        struct sockaddr_in server;

    public:
        tcp_client();
        bool conn(std::string, int);
        bool Send(const void* Data, uint32_t Size);
        size_t Recv(void * buffer, size_t len);
};

tcp_client::tcp_client()
{
    sock = -1;
    port = 0;
    address = "";
}

/**
Connect to a host on a certain port number
*/
bool tcp_client::conn(std::string address , int port)
{
    // create socket if it is not already created
    if(sock == -1)
    {
        //Create socket
        sock = socket(AF_INET , SOCK_STREAM , 0);
        if (sock == -1)
        {
            perror("Could not create socket");
        }

        std::cout<<"Socket created\n";
    }
    else { /* OK , nothing */ }

    //setup address structure
    if(inet_addr(address.c_str()) == -1)
    {
        struct hostent *he;
        struct in_addr **addr_list;

        //resolve the hostname, its not an ip address
        if ( (he = gethostbyname( address.c_str() ) ) == NULL)
        {
            //gethostbyname failed
            std::cout<<"Failed to resolve hostname\n";

            return false;
        }

        //Cast the h_addr_list to in_addr , since h_addr_list also has the ip address in long format only
        addr_list = (struct in_addr **) he->h_addr_list;

        for(int i = 0; addr_list[i] != NULL; i++)
        {
            //strcpy(ip , inet_ntoa(*addr_list[i]) );
            server.sin_addr = *addr_list[i];

            std::cout<<address<<" resolved to "<<inet_ntoa(*addr_list[i])<<std::endl;

            break;
        }
    } else //plain ip address
    {
        server.sin_addr.s_addr = inet_addr( address.c_str() );
    }

    server.sin_family = AF_INET;
    server.sin_port = htons( port );

    //Connect to remote server
    if (connect(sock , (struct sockaddr *)&server , sizeof(server)) < 0)
    {
        perror("connect failed. Error");
        return false;
    }

    int old = fcntl(sock, F_GETFL, 0);
    if (fcntl(sock, F_SETFL, old|O_NONBLOCK) < 0) {

    }

    std::cout<<"Connected\n";
    return true;
}

/**
Send data to the connected host
*/
bool tcp_client::Send(const void* Data, uint32_t Size)
{
    //Send some data
    if( send(sock , Data , Size , 0) < 0)
    {
        perror("Send failed : ");
        return false;
    }

    return true;
}

/**
Receive data from the connected host
*/
size_t tcp_client::Recv(void * buffer, size_t len)
{
    //Receive a reply from the server
    return recv(sock , buffer , len, 0);
}

FUE4Connection::FUE4Connection(IUE4ConnectionObserver& Observer,const std::string& UE4_DRC):
    Observer(Observer),
    bAsAgentServer(true)
{
    mUE4_DRC = UE4_DRC;
    ParseUE4DRC();
    mTCPClient = new tcp_client();
}

FUE4Connection::~FUE4Connection()
{
    cleanupConnection();
    delete mTCPClient;
}

void FUE4Connection::Connect(const std::string& IP, uint16_t Port)
{
    std::thread([this, IP, Port]() //worker thread.
    {
        bool res = mTCPClient->conn(IP, Port);

        if (res) {
            OnConnect();
        } else {
            OnDisconnect(-1);
        }

        std::this_thread::sleep_for(std::chrono::milliseconds(100));
        while (mbUE4Connected) {
            std::this_thread::sleep_for(std::chrono::milliseconds(10));
            do
            {
                int ReceivedBytes = this->mTCPClient->Recv(TmpReadBuffer, sizeof(TmpReadBuffer));

                if (ReceivedBytes <= 0)
                {
                    break;
                }
                ReadBuffer.insert(ReadBuffer.end(), TmpReadBuffer, TmpReadBuffer + ReceivedBytes);
            } while (true);

            uint32_t Consumed = 0;
            while (!ReadBuffer.empty() &&
                (Consumed = this->OnRead(&ReadBuffer.front(), static_cast<uint32_t>(ReadBuffer.size()))))
            {
                ReadBuffer.erase(ReadBuffer.begin(), ReadBuffer.begin() + Consumed);
            }
        }
    }).detach();
}

void FUE4Connection::OnConnect()
{
    std::cout << "FUE4Connection::OnConnect.";
    mbUE4Connected = true;
    Observer.OnUE4Connected();
    StartStreaming();
}

void FUE4Connection::OnDisconnect(int Err)
{
    // EG_LOG(LogDefault, Log, "FUE4Connection::OnDisconnect %d.",Err);
    mbUE4Connected = false;
    Observer.OnUE4Disconnected();
}

void FUE4Connection::StartStreaming()
{
    bStreamingStarted = true;

    const auto msg = EToUE4Msg::StartStreaming;
    Send(&msg, sizeof(msg));
    SetRate(mUE4DRCMap[0].kbps,mUE4DRCMap[0].fps);
}

void FUE4Connection::StopStreaming()
{
    if (!bStreamingStarted) return;

    const auto msg = EToUE4Msg::StopStreaming;
    Send(&msg, sizeof(msg));
    bStreamingStarted = false;
}

void FUE4Connection::ForceKeyFrame()
{
    const auto msg = EToUE4Msg::IFrameRequest;
    Send(&msg, sizeof(msg));
}

void FUE4Connection::SetRate(uint32_t BitrateKbps, uint32_t Framerate)
{
    {
        uint8_t Buf[1 + sizeof(uint16_t)] = {
            static_cast<uint8_t>(EToUE4Msg::AverageBitrateRequest) };
        if (BitrateKbps > std::numeric_limits<uint16_t>::max())
        {
            // EG_LOG(LogDefault, Log, "%s : BitrateKbps is %u . Clamping to 65535.", __FUNCTION__, BitrateKbps);
            BitrateKbps = std::numeric_limits<uint16_t>::max();
        }

        *reinterpret_cast<uint16_t*>(&Buf[1]) = static_cast<uint16_t>(BitrateKbps);
        Send(Buf, sizeof(Buf));
    }

    {
        uint8_t Buf[1 + sizeof(uint8)] = { static_cast<uint8_t>(EToUE4Msg::MaxFpsRequest) };
        Buf[1] = static_cast<uint8>(Framerate);
        Send(Buf, sizeof(Buf));
    }
}

void FUE4Connection::SetVideoResolution(uint32_t Width, uint32_t Height)
{
/*	Json::Reader Reader;
    Json::Value Jmessage;
    if (!Reader.parse("{\"Resolution\":{\"Width\":1280,\"Height\":720}}", Jmessage))
    {
        return;
    }
    Jmessage["Resolution"]["Width"] = Width;
    Jmessage["Resolution"]["Height"] = Height;

    // std::stringstream s;
    // std::string c ;

    // s << "Encoder.TargetSize " << Width << "x" << Height;

    // c = s.str();
    // Jmessage["ConsoleCommand"] = c.c_str();

    int len = Jmessage.toStyledString().length();
    int buf_len = 1+sizeof(uint16_t)+2*len;

    std::string data_buf;
    data_buf.resize(buf_len);

    {
        data_buf[0] = { static_cast<uint8_t>(EToUE4Msg::Command) };
        data_buf[1] = static_cast<uint16_t>(len) & 0xFF;
        data_buf[2] = (static_cast<uint16_t>(len) & 0xFF00) >>8;
        for (int i=0;i <len; i++) { //ASCII -> UTF-16
            data_buf[1+sizeof(uint16_t)+i*2] = Jmessage.toStyledString().c_str()[i];
            data_buf[1+sizeof(uint16_t)+i*2+1] = '\0';
        }
        Send(data_buf.data(), buf_len);
    }*/
}

void FUE4Connection::Send(const void* Data, uint32_t Size)
{
    auto Lk = std::unique_lock<std::mutex>(mMtx);
    mTCPClient->Send(Data, Size);
}

uint32_t FUE4Connection::OnRead(const uint8_t* Data, uint32_t Size)
{
    mbNoDataComing = false;
    if (!bStreamingStarted)
        return Size; // drop data as there's no clients to receive it

    using FTimestamp = uint64_t;
    using FPayloadSize = uint32_t;

    if (Size < sizeof(FTimestamp) + sizeof(EToProxyMsg) + sizeof(FPayloadSize))
        return 0;

    const uint8_t* Ptr = Data;  // pointer to current read pos in the buffer

    auto CaptureTimeMs = *reinterpret_cast<const FTimestamp*>(Ptr);
    Ptr += sizeof(CaptureTimeMs);

    auto PktType = *reinterpret_cast<const EToProxyMsg*>(Ptr);
    Ptr += sizeof(PktType);

    auto PayloadSize = *reinterpret_cast<const FPayloadSize*>(Ptr);
    Ptr += sizeof(PayloadSize);

    if (Ptr + PayloadSize > Data + Size)
        return 0;

    Observer.OnUE4Packet(PktType, Ptr, PayloadSize);

    Ptr += PayloadSize;

    return static_cast<uint32_t>(Ptr - Data);
}

void FUE4Connection::ParseUE4DRC()
{
    int start_pos = 0;
    int end_pos = 0;
    int i = 0;
    // "0:1500:1080x1440:15,10:1000:600x800:15,20:500:600x800:15";
    std::string p = mUE4_DRC;
    memset(mUE4DRCMap,0,sizeof(mUE4DRCMap));
    mUE4DRCMap[0].kbps = 1500;
    mUE4DRCMap[0].fps = 15;

    if (p.length() < 1) return;
    if (p[p.length()-1]!=',') p = p + ",";

    while((end_pos=p.find(",",start_pos))!=std::string::npos)
    {
        mUE4DRCMap[i].nacks = atoi(p.substr(start_pos).c_str());
        start_pos = p.find(":",start_pos)+1;
        mUE4DRCMap[i].kbps = atoi(p.substr(start_pos).c_str());
        start_pos = p.find(":",start_pos)+1;
        mUE4DRCMap[i].width = atoi(p.substr(start_pos).c_str());
        start_pos = p.find("x",start_pos)+1;
        mUE4DRCMap[i].height = atoi(p.substr(start_pos).c_str());
        start_pos = p.find(":",start_pos)+1;
        mUE4DRCMap[i].fps = atoi(p.substr(start_pos).c_str());
        start_pos = end_pos + 1;
        i++;
    }
}

void FUE4Connection::cleanupConnection()
{
    if (mbUE4Connected) {
        mbUE4Connected = false;
        std::this_thread::sleep_for(std::chrono::milliseconds(100));
    }
}