// Copyright 2020 @baidu.com. All rights reserved.

#pragma once

#include <string>
#include <atomic>
#include <mutex>
#include <vector>
#include "ProtocolDefs.h"

class tcp_client;

// callback interface for `FUE4Connection`
struct IUE4ConnectionObserver
{
	virtual ~IUE4ConnectionObserver() {}

	virtual void OnUE4Connected() = 0;
	virtual void OnUE4Disconnected() = 0;
	virtual void OnUE4ConnecttingTimeOut() = 0;
	// reports incoming complete packet from UE4 preserving packet boundaries
	virtual void OnUE4Packet(PixelStreamingProtocol::EToProxyMsg PktType, const void* Pkt, int Size) = 0;
};

typedef struct UE4_DRC_CFG
{
	int nacks;
	int kbps;
	int width;
	int height;
	int fps;
}UE4_DRC_CFG;

// TCP client connection to UE4, manages UE4 <-> Proxy protocol
// automatically reconnects on disconnection
class FUE4Connection
{
public:
	explicit FUE4Connection(IUE4ConnectionObserver& Observer,const std::string& UE4_DRC="0:1500:1080x1440:15");

	~FUE4Connection();

	// connects until succeeded
	void Connect(const std::string& IP, uint16_t Port);

	// messages to UE4
	void StartStreaming();
	void StopStreaming();
	void ForceKeyFrame();
	void SetRate(uint32_t BitrateKbps, uint32_t Framerate);
	void SetVideoResolution(uint32_t Width, uint32_t Height);
	// generic send for passing messages received from clients
	void Send(const void* Data, uint32_t Size);

	void setUE4_DRC(const char *ue4drc) { mUE4_DRC = ue4drc; ParseUE4DRC();}

	void cleanupConnection();

	void OnConnect() ;
	uint32_t OnRead(const uint8_t* Data, uint32_t Size) ;
	void OnDisconnect(int Err) ;


private:
	void ParseUE4DRC();

private:
	IUE4ConnectionObserver& Observer;
	std::atomic<bool> bStreamingStarted = {false};
	std::atomic<bool> bAsAgentServer = {false};
	std::atomic<bool> mbNoDataComing = {false};
	std::atomic<bool> mbNoListenerNACKComing = {true};
	std::atomic<bool> mbUE4Connected = {false};
	UE4_DRC_CFG mUE4DRCMap[10];
	std::mutex mMtx;
	std::string mUE4_DRC = "";
	tcp_client  *mTCPClient;

	uint8_t TmpReadBuffer[0xFFFF];
	std::vector<uint8_t> ReadBuffer;
};
