// Copyright 2020 shaohongsheng@baidu.com. All rights reserved.

#pragma once

#include <string>
#include <locale>
#include <vector>
#include <algorithm>
#include <cstring>

namespace detail
{
	// templated version of my_equal so it could work with both char and wchar_t
	template <typename charT>
	struct TCharEqual
	{
		TCharEqual(const std::locale& loc)
		    : loc_(loc)
		{
		}
		bool operator()(charT ch1, charT ch2) { return std::toupper(ch1, loc_) == std::toupper(ch2, loc_); }

	  private:
		const std::locale& loc_;
	};
}

/**
 * Search for a substring (case insensitive)
 * @param Where String to search in
 * @param What string to search for in "Where"
 * @return Position where the substring was found, or -1 if not found.
 */
// find substring (case insensitive)
template <typename T>
static int CiFindSubStr(const T& Where, const T& What, const std::locale& loc = std::locale())
{
	typename T::const_iterator It =
	    std::search(Where.begin(), Where.end(), What.begin(), What.end(), detail::TCharEqual<typename T::value_type>(loc));
	if (It != Where.end())
		return It - Where.begin();
	else
		return -1;  // not found
}

/**
 * Checks if two strings are equal (case insensitive)
 */
template <typename T>
static bool CiEquals(const T& Str1, const T& Str2, const std::locale& loc = std::locale())
{
	if (Str1.size() != Str2.size())
		return false;
	typename T::const_iterator It1 = Str1.begin();
	typename T::const_iterator It2 = Str2.begin();
	detail::TCharEqual<typename T::value_type> Eq(loc);
	while (It1 != Str1.end())
	{
		if (!Eq(*It1, *It2))
			return false;
		++It1;
		++It2;
	}
	return true;
}


/**
 * Utility class to parse command line parameters.
 *
 * Arguments need to be prefixed with '-' and can have the following formats:
 * -SomeArg
 * -SomeOtherArg=Value
 */
class FCmdLine
{
public:
	struct FParam
	{
		template <class T1, class T2>
		FParam(T1&& Name, T2&& Value)
		    : Name(std::forward<T1>(Name))
		    , Value(std::forward<T2>(Value))
		{
		}
		std::string Name;
		std::string Value;
	};

	FCmdLine()
	{
	}

	/**
	 * Parse all the supplied parameters, as received in "main"
	 */
	bool Parse(int Argc, char* Argv[], bool CaseSensitive=false);

	/**
	 * Checks if the specified parameter is present, in any acceptable form, such
	 * ash "-arg" or "-arg=value"
	 */
	bool Has(const char* Name) const;

	/**
	 * Gets the value of the specified parameter.
	 * @return Value of the parameter or an empty string if the parameter doesn't
	 * exist or doesn't is not in the "-arg=value" form
	 * Use "Has" method first, to check if a parameter exists.
	 */
	const std::string& Get(const char* Name) const;

	/**
	 * Gets the value of the specified parameter, as an integer
	 *
	 * @param Name Parameter name
	 * @param DefaultValue If the parameter doesn't exist or is not in the "-arg=value" form, it will default to this
	 * @return
	 *	Pair where "first" is true if the parameter exists, false if not. "second" is the parameter's value or DefaultValue
	 */
	std::pair<bool, int> GetAsInt(const char* Name, int DefaultValue = 0) const;

	/**
	 * @return The number of parameters
	 */
	int GetCount() const;

private:
	bool Equals(const std::string& A, const char* B) const;

	std::vector<FParam> Params;
	static std::string Empty;
	bool CaseSensitive = true;
};
