#!/usr/bin/env python
"""Chat using realtime API and SoundDevice."""
# -*- coding: utf-8 -*-

import asyncio
import json
import sys
import sounddevice as sd
from websockets.asyncio.client import connect

# =================配置区域=================
# 百度 RTC 参数
RATE = 16000
CHANNELS = 1
DTYPE = 'int16'  # 16位深度
CHUNK = 3200  # 每次读写的数据块大小

# 授权信息 (替换为真实Key)
LIC_KEY = "lickey"
DEV_ID = "myDev"
USER_ID = "User01"

speaking = False

# ==========================================

async def send_audio_loop(websocket, stream_in, loop):
    """发送端：使用 sounddevice 读取麦克风"""
    print(">>> 正在监听麦克风...")

    def read_mic():
        # sounddevice 的 read 是阻塞的，读取指定数量的帧
        # 返回值是 (data, overflow_flag)
        data, _ = stream_in.read(CHUNK)
        return bytes(data)  # 转为 bytes 发送

    try:
        while True:
            # 放到线程池运行，避免阻塞
            pcm_bytes = await loop.run_in_executor(None, read_mic)
            if speaking:
                continue
            await websocket.send(pcm_bytes)
    except Exception as e:
        print(f"发送循环停止: {e}")


async def receive_loop(websocket, stream_out, loop):
    """接收端"""
    print(">>> 准备接收回复...")
    
    global speaking

    def play_pcm(data):
        stream_out.write(data)

    try:
        while True:
            msg = await websocket.recv()

            if isinstance(msg, str):
                # 1. 鉴权
                if msg.startswith("[E]:[LIC]:[MUST]"):
                    print("[系统] 发送授权...")
                    lic = f'[E]:[LIC]:[ACTIVE]:{{"devId":"{DEV_ID}","uId":"{USER_ID}","licKey":"{LIC_KEY}"}}'
                    await websocket.send(lic)

                # 2. 识别到的用户语音
                elif msg.startswith("[Q]:"):
                    if not msg.startswith("[Q]:[M]:"):
                        print(f"--------\n问: {msg[4:]}")

                # 3. AI 的回答文本
                elif msg.startswith("[A]:"):
                    print(f"AI答: {msg[4:]}")

                # 4. 【关键】识别到的意图/Function Call
                # 这里的格式通常是 [F]:[C]:{json数据}
                elif msg.startswith("[F]:[C]:"):
                    # 去掉前缀 [F]:[C]: 拿到后面的 JSON 字符串
                    json_str = msg[8:]
                    try:
                        data = json.loads(json_str)
                        # 打印原始数据方便调试
                        print(f"\n★ 命中意图 (Function Call) ★")
                        print(f"原始数据: {json_str}")

                        # 解析具体字段 (根据你的截图配置)
                        if "fname" in data and data["fname"] == "add_clock":
                            params = data.get("param", {})
                            time_val = params.get("time", "未知时间")
                            date_val = params.get("date", "未知日期")
                            print(f"指令: 添加闹钟")
                            print(f" -> 时间: {time_val}")
                            print(f" -> 日期: {date_val}")

                            # 在这里可以写代码真的去设置电脑闹钟
                            # ...

                    except json.JSONDecodeError:
                        print(f"收到FC指令但解析JSON失败: {json_str}")
                        
                elif msg.startswith('[E]:[TTS_BEGIN_SPEAKING]'):
                    speaking = True
                elif msg.startswith('[E]:[TTS_END_SPEAKING]'):
                    speaking = False
                else :
                    print(f"RX: {msg}")

            elif isinstance(msg, bytes):
                if len(msg) > 0:
                    await loop.run_in_executor(None, play_pcm, msg)

    except Exception as e:
        print(f"接收循环停止: {e}")

async def run_chat(uri):
    """Chat Main."""
    print("正在打开音频设备 (SoundDevice)...")

    # 打开原始输入流 (RawInputStream) 处理 bytes
    # 打开原始输出流 (RawOutputStream) 处理 bytes
    with sd.RawInputStream(samplerate=RATE, channels=CHANNELS, dtype=DTYPE, blocksize=CHUNK) as stream_in, \
            sd.RawOutputStream(samplerate=RATE, channels=CHANNELS, dtype=DTYPE, blocksize=CHUNK) as stream_out:
        print("音频设备就绪。")
        loop = asyncio.get_running_loop()

        async with connect(uri) as websocket:
            print(">> 已连接到", uri)

            send_task = asyncio.create_task(send_audio_loop(websocket, stream_in, loop))
            recv_task = asyncio.create_task(receive_loop(websocket, stream_out, loop))

            done, pending = await asyncio.wait([send_task, recv_task], return_when=asyncio.FIRST_COMPLETED)
            for t in pending:
                t.cancel()


if __name__ == "__main__":
    try:
        uri = "ws://rtc-aiotgw.exp.bcelive.com/v1/realtime?a=APPID&ak=AK&sk=SK"
        if len(sys.argv) > 1:
            uri = sys.argv[1]

        if sys.platform == 'win32':
            asyncio.set_event_loop_policy(asyncio.WindowsSelectorEventLoopPolicy())

        asyncio.run(run_chat(uri))
    except KeyboardInterrupt:
        print("\n已退出")
    except Exception as e:
        print(f"\n发生错误: {e}")
