//
//  BaiduRtcLastmile.h
//  rtc_sdk_ios
//
//  Created by Sun,Jian(ACU-T102) on 2018/12/5.
//  Copyright © 2018年 Sun,Jian(ACU-T102). All rights reserved.
//

#ifndef BaiduRtcLastmile_h
#define BaiduRtcLastmile_h
#import <BaiduRtcRoom/BaiduRtcRoomApiDefines.h>

@protocol BaiduRtcLastmileDelegate;

/** The state of the probe test result. */
typedef NS_ENUM(NSUInteger, RtcLastmileResultState) {
    /** 1: the last-mile network probe test is complete. */
    RtcLastmileResultComplete = 1,
    /** 2: the last-mile network probe test is incomplete and the bandwidth estimation is not available, probably due to limited test resources. */
    RtcLastmileResultIncompleteNoBwe = 2,
    /** 3: the last-mile network probe test is not carried out, probably due to poor network conditions. */
    RtcLastmileResultUnavailable = 3,
};

/** Network quality. */
typedef NS_ENUM(NSUInteger, RtcLastmileQuality) {
    /** The network quality is unknown. */
    RtcLastmileQualityUnknown = 0,
    /**  The network quality is excellent. */
    RtcLastmileQualityExcellent = 1,
    /** The network quality is quite good, but the bitrate may be slightly lower than excellent. */
    RtcLastmileQualityGood = 2,
    /** Users can feel the communication slightly impaired. */
    RtcLastmileQualityPoor = 3,
    /** Users can communicate only not very smoothly. */
    RtcLastmileQualityBad = 4,
     /** The network quality is so bad that users can hardly communicate. */
    RtcLastmileQualityVBad = 5,
     /** The network is disconnected and users cannot communicate at all. */
    RtcLastmileQualityDown = 6,
     /** Users cannot detect the network quality. (Not in use.) */
    RtcLastmileQualityUnsupported = 7,
     /** Detecting the network quality. */
    RtcLastmileQualityDetecting = 8,
};

/** The last-mile network probe test result in one direction (uplink or downlink). */
@interface RtcLastmileOneWayResult : NSObject
/** The packet loss rate (%). */
@property (assign, nonatomic) NSUInteger lossRate;
/** The network jitter (ms). */
@property (assign, nonatomic) NSUInteger jitter;
/** The estimated available bandwidth (Kbps). */
@property (assign, nonatomic) NSUInteger bandwidth;
@end

@interface RtcLastmileResult : NSObject
/** The state of the probe test. See RtcLastmileResultState */
@property (assign, nonatomic) RtcLastmileResultState state;
/** The round-trip delay time (ms). */
@property (assign, nonatomic) NSUInteger rtt;
/** Network quality. see RtcLastmileQuality */
@property (assign, nonatomic) NSUInteger quality;
/** The uplink last-mile network report */
@property (strong, nonatomic) RtcLastmileOneWayResult *_Nonnull uplinkReport;
/** The downlink last-mile network report */
@property (strong, nonatomic) RtcLastmileOneWayResult *_Nonnull downlinkReport;
@end

@interface BaiduRtcLastmile : NSObject

/**
 初始化 Lastmile
 
 @param appId 百度 派发的数字 ID, 开发者的唯一标识
 @param tokenStr app server 派发的token字符串, 用来校验对应signal 通信的合法性
 @param uplinkBitrate  码率
 @param downlinkBitrate  码率
 @return Lastmile 实例，nil 表示初始化失败
 @discussion 初始化 Lastmile 时调用。初始化 Lastmile 失败可能导致 Lastmile 功能异常
 */
- (instancetype)initProbeTest:(NSString *)appId
                     tokenStr:(NSString *)tokenStr
                uplinkBitrate:(int)uplinkBitrate
              downlinkBitrate:(int)downlinkBitrate
                     delegate:(id<BaiduRtcLastmileDelegate>)delegate;

/**
 * 结束网络探测, 可以在网络探测回调函数onLastmileProbeResult()中调用, 也可以手动触发结束
 *
 * @discussion 在探测结果还未触发前结束网络探测将不会报告探测结果
 */
- (void)stopProbeTest;

/**
 * 设置网络探测服务器地址, 默认不需要调用该函数
 *
 * @param serverUrl 探测服务器的地址
 */
+ (void)setServerUrl:(NSString *)serverUrl;

@end

#pragma mark -- 提供app的callback
@protocol BaiduRtcLastmileDelegate <NSObject>

- (void)rtcLastmile:(BaiduRtcLastmile *)lastmile lastmileProbeTestResult:(RtcLastmileResult *)result;

@end

#endif /* BaiduRtcLastmile_h */
