//
//  BCAAudioRingBuffer.m
//  ChatAgentApp
//
//  Created by caoshiqian on 2025/8/27.
//

#import "BCAAudioRingBuffer.h"
#import <pthread.h>
#import <string.h>
#import <os/lock.h>

@interface BCAAudioRingBuffer () {
    uint8_t *_buffer;
    size_t _capacity;
    size_t _readPos;
    size_t _writePos;
    size_t _bytesAvailable;
    pthread_mutex_t _mutex;
}

@end

@implementation BCAAudioRingBuffer

- (instancetype)initWithCapacity:(size_t)capacity {
    self = [super init];
    if (self) {
        _capacity = capacity;
        _buffer = (uint8_t *)malloc(_capacity);
        _readPos = 0;
        _writePos = 0;
        _bytesAvailable = 0;
        
        // 初始化互斥锁属性
        pthread_mutexattr_t attr;
        pthread_mutexattr_init(&attr);
        pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_NORMAL); // 使用普通互斥锁
        
        // 初始化互斥锁
        pthread_mutex_init(&_mutex, &attr);
        
        // 销毁属性
        pthread_mutexattr_destroy(&attr);
        
        if (_buffer == NULL) {
            return nil;
        }
    }
    return self;
}

- (void)dealloc {
    if (_buffer) {
        free(_buffer);
        _buffer = NULL;
    }
    // 销毁互斥锁
    pthread_mutex_destroy(&_mutex);
}

#pragma mark - Public Properties

- (size_t)bytesAvailable {
    pthread_mutex_lock(&_mutex);
    size_t available = _bytesAvailable;
    pthread_mutex_unlock(&_mutex);
    return available;
}

- (size_t)spaceAvailable {
    pthread_mutex_lock(&_mutex);
    size_t space = _capacity - _bytesAvailable;
    pthread_mutex_unlock(&_mutex);
    return space;
}

#pragma mark - Public Methods

- (size_t)write:(const void *)data length:(size_t)length {
    if (data == NULL || length == 0) {
        return 0;
    }
    
    pthread_mutex_lock(&_mutex);
    
    size_t bytesToWrite = MIN(length, _capacity - _bytesAvailable);
    if (bytesToWrite == 0) {
        pthread_mutex_unlock(&_mutex);
        return 0;
    }
    
    size_t bytesToEnd = _capacity - _writePos;
    size_t firstPart = MIN(bytesToWrite, bytesToEnd);
    size_t secondPart = bytesToWrite - firstPart;
    
    // 写入第一部分数据（到缓冲区末尾）
    if (firstPart > 0) {
        memcpy(_buffer + _writePos, data, firstPart);
    }
    
    // 写入第二部分数据（从缓冲区开头开始）
    if (secondPart > 0) {
        memcpy(_buffer, (const uint8_t *)data + firstPart, secondPart);
    }
    
    // 更新写位置
    _writePos = (_writePos + bytesToWrite) % _capacity;
    
    // 更新可用数据量
    _bytesAvailable += bytesToWrite;
    
    pthread_mutex_unlock(&_mutex);
    
    return bytesToWrite;
}

- (size_t)read:(void *)data length:(size_t)length {
    if (data == NULL || length == 0) {
        return 0;
    }
    
    pthread_mutex_lock(&_mutex);
    
    size_t bytesToRead = MIN(length, _bytesAvailable);
    if (bytesToRead == 0) {
        pthread_mutex_unlock(&_mutex);
        return 0;
    }
    
    size_t bytesToEnd = _capacity - _readPos;
    size_t firstPart = MIN(bytesToRead, bytesToEnd);
    size_t secondPart = bytesToRead - firstPart;
    
    // 读取第一部分数据（从读位置到缓冲区末尾）
    if (firstPart > 0) {
        memcpy(data, _buffer + _readPos, firstPart);
    }
    
    // 读取第二部分数据（从缓冲区开头开始）
    if (secondPart > 0) {
        memcpy((uint8_t *)data + firstPart, _buffer, secondPart);
    }
    
    // 更新读位置
    _readPos = (_readPos + bytesToRead) % _capacity;
    
    // 更新可用数据量
    _bytesAvailable -= bytesToRead;
    
    pthread_mutex_unlock(&_mutex);
    
    return bytesToRead;
}

- (size_t)peek:(void *)data length:(size_t)length {
    if (data == NULL || length == 0) {
        return 0;
    }
    
    pthread_mutex_lock(&_mutex);
    
    size_t bytesToRead = MIN(length, _bytesAvailable);
    if (bytesToRead == 0) {
        pthread_mutex_unlock(&_mutex);
        return 0;
    }
    
    size_t bytesToEnd = _capacity - _readPos;
    size_t firstPart = MIN(bytesToRead, bytesToEnd);
    size_t secondPart = bytesToRead - firstPart;
    
    // 读取第一部分数据（从读位置到缓冲区末尾）
    if (firstPart > 0) {
        memcpy(data, _buffer + _readPos, firstPart);
    }
    
    // 读取第二部分数据（从缓冲区开头开始）
    if (secondPart > 0) {
        memcpy((uint8_t *)data + firstPart, _buffer, secondPart);
    }
    
    pthread_mutex_unlock(&_mutex);
    
    return bytesToRead;
}

- (void)clear {
    pthread_mutex_lock(&_mutex);
    
    _readPos = 0;
    _writePos = 0;
    _bytesAvailable = 0;
    memset(_buffer, 0, _capacity);
    
    pthread_mutex_unlock(&_mutex);
}

@end
