//
//  BCAAudioExternalDevice.m
//  ChatAgentApp
//
//  Created by caoshiqian on 2025/8/27.
//

#import "BCAAudioExternalDevice.h"
#import <AVFoundation/AVFoundation.h>
#import "BCAAudioRingBuffer.h"

#define AudioLog(format, ...) do { \
    NSString *type = (self.deviceType == BCAAudioDeviceTypeRecord) ? @"Record" : @"Player"; \
    NSLog(@"[AgentAudio][%@] " format, type, ##__VA_ARGS__); \
} while (0)

#define CHECK_AUDIO_UNIT() \
    if (!_audioUnit) { \
        AudioLog(@"Audio unit not initialized"); \
        return NO; \
    } \

#define kOutputBus 0
#define kInputBus 1

#define kPrefferedSampleRate 16000

static void LogStreamDescription(AudioStreamBasicDescription description) {
  char formatIdString[5];
  UInt32 formatId = CFSwapInt32HostToBig(description.mFormatID);
  bcopy(&formatId, formatIdString, 4);
  formatIdString[4] = '\0';
  NSLog(@"AudioStreamBasicDescription: {\n"
          "  mSampleRate: %.2f\n"
          "  formatIDString: %s\n"
          "  mFormatFlags: 0x%X\n"
          "  mBytesPerPacket: %u\n"
          "  mFramesPerPacket: %u\n"
          "  mBytesPerFrame: %u\n"
          "  mChannelsPerFrame: %u\n"
          "  mBitsPerChannel: %u\n"
          "  mReserved: %u\n}",
         description.mSampleRate, formatIdString,
         description.mFormatFlags,
         description.mBytesPerPacket,
         description.mFramesPerPacket,
         description.mBytesPerFrame,
         description.mChannelsPerFrame,
         description.mBitsPerChannel,
         description.mReserved);
}

@interface BCAAudioExternalDeviceImpl () {
    AudioComponentInstance _audioUnit;
    BOOL _isRecording;
    BOOL _isPlaying;
    BCAAudioRingBuffer *_buffer;
}

@property (nonatomic, weak) id<BaiduRtcRoomApiAudioExternalRecordClient> recordClient;
@property (nonatomic, weak) id<BaiduRtcRoomApiAudioExternalPlayerClient> playerClient;

@property (nonatomic, assign) AudioStreamBasicDescription audioFormat;
@property (nonatomic, assign) double sampleRate;
@property (nonatomic, assign) UInt32 channels;
@property (nonatomic, strong) dispatch_queue_t audioQueue;

@end

@implementation BCAAudioExternalDeviceImpl

#pragma mark - Initialization

- (instancetype)initWithType:(BCAAudioDeviceType)type {
    self = [super init];
    if (self) {
        _deviceType = type;
        _audioQueue = dispatch_queue_create("com.baidu.rtcagent.audio.queue", DISPATCH_QUEUE_SERIAL);
        _sampleRate = kPrefferedSampleRate;
        _channels = 1;
        _isRecording = NO;
        _isPlaying = NO;
        
        // 2s buffer for recording
        size_t bufferSize = _sampleRate * _channels * sizeof(SInt16) * 2;
        _buffer = [[BCAAudioRingBuffer alloc] initWithCapacity:bufferSize];
        
        [self setupAudioFormat];
        [self initAudioComponent];
    }
    return self;
}

- (void)dealloc {
    [self disposeAudioUnit];
}

#pragma mark - Audio Config

- (void)setupAudioFormat {
    _audioFormat.mSampleRate = _sampleRate;
    _audioFormat.mFormatID = kAudioFormatLinearPCM;
    _audioFormat.mFormatFlags = kAudioFormatFlagIsSignedInteger | kAudioFormatFlagIsPacked;
    _audioFormat.mChannelsPerFrame = _channels;
    _audioFormat.mFramesPerPacket = 1;
    _audioFormat.mBitsPerChannel = 16;
    _audioFormat.mBytesPerFrame = (_audioFormat.mBitsPerChannel / 8) * _audioFormat.mChannelsPerFrame;
    _audioFormat.mBytesPerPacket = _audioFormat.mBytesPerFrame * _audioFormat.mFramesPerPacket;
}

#pragma mark - BaiduRtcRoomApiAudioExternalRecordDevice Protocol

- (BOOL)initializeExternalRecordFormat:(AudioStreamBasicDescription *)asbd {
    if (asbd) {
        memcpy(&_audioFormat, asbd, sizeof(AudioStreamBasicDescription));
    }
    
    LogStreamDescription(*asbd);
    
    OSStatus status = [self setupAudioUnit];
    if (status != noErr) {
        AudioLog(@"Failed to setup audio unit: %d", (int)status);
        return NO;
    }
    
    AudioLog(@"Audio unit initialized");
    return YES;
}

- (BOOL)uninitializeExternalRecord {
    _isRecording = NO;
    BOOL ret = [self uninitAudioUnit];
    return ret;
}

- (BOOL)startExternalRecord {
    BOOL ret = [self startAudioUnit];
    if (ret) {
        _isRecording = YES;
    }
    return ret;
}

- (BOOL)stopExternalRecord {
    BOOL ret = [self stopAudioUnit];
    if (ret) {
        _isRecording = NO;
    }
    return ret;
}

- (OSStatus)onRenderData:(AudioUnitRenderActionFlags *)flags
                    time:(const AudioTimeStamp *)time
                  busNum:(UInt32)busNum
               numFrames:(UInt32)numFrames
                  ioData:(AudioBufferList *)ioData {
    
    UInt32 bytesNeeded = numFrames * _audioFormat.mBytesPerFrame;
    
    // 从 buffer 中读取采集数据，填充到 SDK 需要的缓存结构中进行渲染
    for (UInt32 i = 0; i < ioData->mNumberBuffers; i++) {
        AudioBuffer *buffer = &ioData->mBuffers[i];
        size_t bytesRead = [_buffer read:buffer->mData length:bytesNeeded];
        
        if (bytesRead < bytesNeeded) {
            // 缓冲区数据不足，用静音填充剩余部分
            UInt32 silenceBytes = (UInt32)(bytesNeeded - bytesRead);
            memset((UInt8 *)buffer->mData + bytesRead, 0, silenceBytes);
            
            AudioLog(@"buffer underrun: needed %u bytes, got %zu bytes", bytesNeeded, bytesRead);
        }
    }
    
    return noErr;
}

#pragma mark - BaiduRtcRoomApiAudioExternalPlayerDevice Protocol

- (BOOL)initializeExternalPlayerFormat:(AudioStreamBasicDescription *)asbd {
    if (asbd) {
        memcpy(&_audioFormat, asbd, sizeof(AudioStreamBasicDescription));
    }
    
    LogStreamDescription(*asbd);
    
    OSStatus status = [self setupAudioUnit];
    if (status != noErr) {
        AudioLog(@"Failed to setup audio unit: %d", (int)status);
        return NO;
    }
    
    AudioLog(@"Audio unit initialized");
    return YES;
}

- (BOOL)uninitializeExternalPlayer {
    _isPlaying = NO;
    BOOL ret = [self uninitAudioUnit];
    return ret;
}

- (BOOL)startExternalPlayer {
    BOOL ret = [self startAudioUnit];
    if (ret) {
        _isPlaying = YES;
    }
    return ret;
}

- (BOOL)stopExternalPlayer {
    BOOL ret = [self stopAudioUnit];
    if (ret) {
        _isPlaying = NO;
    }
    return YES;
}

#pragma mark - Audio Unit

- (OSStatus)initAudioComponent {
    // 描述音频组件
    AudioComponentDescription desc;
    desc.componentType = kAudioUnitType_Output;
    desc.componentSubType = kAudioUnitSubType_VoiceProcessingIO;
    desc.componentManufacturer = kAudioUnitManufacturer_Apple;
    desc.componentFlags = 0;
    desc.componentFlagsMask = 0;
    
    // 获取音频组件
    AudioComponent component = AudioComponentFindNext(NULL, &desc);
    if (!component) {
        AudioLog(@"AudioComponentFindNext failed");
        return -1;
    }
    
    // 创建音频单元实例
    OSStatus status = AudioComponentInstanceNew(component, &_audioUnit);
    if (status != noErr) {
        AudioLog(@"AudioComponentInstanceNew failed: %d", (int)status);
        return status;
    }
    
    return noErr;
}

- (OSStatus)setupAudioUnit {
    BCAAudioDeviceType type = self.deviceType;
    AudioStreamBasicDescription asbd = self.audioFormat;
    OSStatus status = noErr;
    
    if (type == BCAAudioDeviceTypeRecord) {
        // 设置IO
        UInt32 enable = 1;
        status = AudioUnitSetProperty(_audioUnit, kAudioOutputUnitProperty_EnableIO,
                                      kAudioUnitScope_Input, kInputBus, &enable, sizeof(enable));
        enable = 0;
        status = AudioUnitSetProperty(_audioUnit, kAudioOutputUnitProperty_EnableIO,
                                      kAudioUnitScope_Output, kOutputBus, &enable, sizeof(enable));
        if (status != noErr) {
            AudioLog(@"EnableIO failed: %d", (int)status);
            return status;
        }
        
        UInt32 flag = 0;
        status = AudioUnitSetProperty(_audioUnit, kAudioUnitProperty_ShouldAllocateBuffer,
                                      kAudioUnitScope_Output, kInputBus, &flag, sizeof(flag));
        if (status != noErr) {
            AudioLog(@"Set ShouldAllocateBuffer failed: %d", (int)status);
            return status;
        }
        
        // 设置音频格式
        status = AudioUnitSetProperty(_audioUnit, kAudioUnitProperty_StreamFormat,
                                      kAudioUnitScope_Output, kInputBus, &asbd, sizeof(asbd));
        if (status != noErr) {
            AudioLog(@"Set stream format failed: %d", (int)status);
            return status;
        }
        
        // 设置渲染回调
        AURenderCallbackStruct callback;
        callback.inputProc = recordingCallback;
        callback.inputProcRefCon = (__bridge void *)self;
        status = AudioUnitSetProperty(_audioUnit, kAudioOutputUnitProperty_SetInputCallback,
                                      kAudioUnitScope_Global, kInputBus, &callback, sizeof(callback));
        if (status != noErr) {
            AudioLog(@"Set callback failed: %d", (int)status);
            return status;
        }
        
    } else if (type == BCAAudioDeviceTypePlayer) {
        // 设置IO
        UInt32 enable = 1;
        status = AudioUnitSetProperty(_audioUnit, kAudioOutputUnitProperty_EnableIO,
                                      kAudioUnitScope_Output, kOutputBus, &enable, sizeof(enable));
        if (status != noErr) {
            AudioLog(@"EnableIO failed: %d", (int)status);
            return status;
        }
        
        // 设置输出格式
        status = AudioUnitSetProperty(_audioUnit, kAudioUnitProperty_StreamFormat,
                                      kAudioUnitScope_Input, kOutputBus, &asbd, sizeof(asbd));
        if (status != noErr) {
            AudioLog(@"Set stream format failed: %d", (int)status);
            return status;
        }
        
        // 设置播放回调
        AURenderCallbackStruct callback;
        callback.inputProc = playbackCallback;
        callback.inputProcRefCon = (__bridge void *)self;
        status = AudioUnitSetProperty(_audioUnit, kAudioUnitProperty_SetRenderCallback,
                                      kAudioUnitScope_Input, kOutputBus, &callback, sizeof(callback));
        if (status != noErr) {
            AudioLog(@"Set callback failed: %d", (int)status);
            return status;
        }
    }
    
    // 初始化音频单元
    status = AudioUnitInitialize(_audioUnit);
    if (status != noErr) {
        [self disposeAudioUnit];
        AudioLog(@"AudioUnitInitialize failed: %d", (int)status);
        return status;
    }
    
    return noErr;
}

- (BOOL)startAudioUnit {
    CHECK_AUDIO_UNIT()
    OSStatus status = AudioOutputUnitStart(_audioUnit);
    AudioLog(@"started: %d", (int)status);
    return status == noErr;
}

- (BOOL)stopAudioUnit {
    CHECK_AUDIO_UNIT()
    OSStatus status = AudioOutputUnitStop(_audioUnit);
    AudioLog(@"stopped: %d", (int)status);
    return status == noErr;
}

- (BOOL)uninitAudioUnit {
    CHECK_AUDIO_UNIT()
    OSStatus status = AudioUnitUninitialize(_audioUnit);
    AudioLog(@"Audio unit uninitialized: %@", @(status));
    return status == noErr;
}

- (BOOL)disposeAudioUnit {
    if (_audioUnit) {
        AudioOutputUnitStop(_audioUnit);
        AudioUnitUninitialize(_audioUnit);
        AudioComponentInstanceDispose(_audioUnit);
        _audioUnit = NULL;
    }
    return YES;
}

#pragma mark - Static Callback Functions

static OSStatus recordingCallback(void *inRefCon,
                                 AudioUnitRenderActionFlags *ioActionFlags,
                                 const AudioTimeStamp *inTimeStamp,
                                 UInt32 inBusNumber,
                                 UInt32 inNumberFrames,
                                 AudioBufferList *ioData) {
    BCAAudioExternalDeviceImpl *self = (__bridge BCAAudioExternalDeviceImpl *)inRefCon;
    
    AudioBufferList bufferList;
    bufferList.mNumberBuffers = 1;
    bufferList.mBuffers[0].mDataByteSize = inNumberFrames * self.audioFormat.mBytesPerFrame;
    bufferList.mBuffers[0].mNumberChannels = self.audioFormat.mChannelsPerFrame;
    bufferList.mBuffers[0].mData = malloc(bufferList.mBuffers[0].mDataByteSize);
    
    // 渲染音频数据
    OSStatus status = AudioUnitRender(self->_audioUnit,
                                     ioActionFlags,
                                     inTimeStamp,
                                     inBusNumber,
                                     inNumberFrames,
                                     &bufferList);
    
    if (status == noErr && self.recordClient) {
        [self->_buffer write:bufferList.mBuffers[0].mData
                      length:bufferList.mBuffers[0].mDataByteSize];
        
        // 通知 SDK 有采集数据
        [self.recordClient onRecordData:ioActionFlags
                                   time:inTimeStamp
                                 busNum:inBusNumber
                              numFrames:inNumberFrames
                                 ioData:&bufferList];
    }
    
    free(bufferList.mBuffers[0].mData);
    return status;
}

static OSStatus playbackCallback(void *inRefCon,
                                AudioUnitRenderActionFlags *ioActionFlags,
                                const AudioTimeStamp *inTimeStamp,
                                UInt32 inBusNumber,
                                UInt32 inNumberFrames,
                                AudioBufferList *ioData) {
    BCAAudioExternalDeviceImpl *self = (__bridge BCAAudioExternalDeviceImpl *)inRefCon;
    
    // 确保有播放客户端并且正在播放
    if (!self.playerClient || !self->_isPlaying) {
        for (UInt32 i = 0; i < ioData->mNumberBuffers; i++) {
            memset(ioData->mBuffers[i].mData, 0, ioData->mBuffers[i].mDataByteSize);
        }
        return noErr;
    }
    
    // 调用 SDK 通过 PlayerClient 协议提供的回调，让 SDK 填充要播放的音频数据
    OSStatus status = [self.playerClient onGetPlayoutData:ioActionFlags
                                                    time:inTimeStamp
                                                  busNum:inBusNumber
                                               numFrames:inNumberFrames
                                                  ioData:ioData];
    
    return status;
}

@end

@interface BCAAudioExternalDevice ()

@property (nonatomic, strong) BCAAudioExternalDeviceImpl *recordDeviceImpl;
@property (nonatomic, strong) BCAAudioExternalDeviceImpl *playerDeviceImpl;

@end

@implementation BCAAudioExternalDevice

- (instancetype)init {
    if (self = [super init]) {
        [self setupAudioSession];
    }
    return self;
}

- (id<BaiduRtcRoomApiAudioExternalRecordDevice>)createExternalRecord:(id<BaiduRtcRoomApiAudioExternalRecordClient>)client {
    self.recordDeviceImpl = [[BCAAudioExternalDeviceImpl alloc] initWithType:BCAAudioDeviceTypeRecord];
    self.recordDeviceImpl.recordClient = client;
    return self.recordDeviceImpl;
}

- (id<BaiduRtcRoomApiAudioExternalPlayerDevice>)createExternalPlayer:(id<BaiduRtcRoomApiAudioExternalPlayerClient>)client {
    self.playerDeviceImpl = [[BCAAudioExternalDeviceImpl alloc] initWithType:BCAAudioDeviceTypePlayer];
    self.playerDeviceImpl.playerClient = client;
    return self.playerDeviceImpl;
}

- (void)setupAudioSession {
    NSError *error = nil;
    AVAudioSession *session = [AVAudioSession sharedInstance];
    
    AVAudioSessionCategory category = AVAudioSessionCategoryPlayAndRecord;
    
    AVAudioSessionCategoryOptions options = AVAudioSessionCategoryOptionDefaultToSpeaker;
    options |= AVAudioSessionCategoryOptionMixWithOthers;
    options |= AVAudioSessionCategoryOptionAllowBluetooth;
    
    [session setCategory:category withOptions:options error:&error];
    [session setMode:AVAudioSessionModeVoiceChat error:&error];
    
    // Sometimes category options don't stick after setting mode.
    if (session.categoryOptions != options) {
        [session setCategory:category withOptions:options error:&error];
    }
    
    if (error) {
        NSLog(@"Failed to set audio session category: %@", error);
        return;
    }
    
    [session setPreferredOutputNumberOfChannels:1 error:&error];
    if (error) {
        NSLog(@"Failed to set preferred channels: %@", error);
    }
    
    [session setPreferredSampleRate:kPrefferedSampleRate error:&error];
    if (error) {
        NSLog(@"Failed to set preferred sample rate: %@", error);
    }
    
    [session setPreferredIOBufferDuration:0.02 error:&error];
    if (error) {
        NSLog(@"Failed to set preferred buffer duration: %@", error);
    }
    
    [session setActive:YES error:&error];
    if (error) {
        NSLog(@"Failed to activate audio session: %@", error);
    }
}

@end
