//
//  AIChatInputBar.m
//  ChatAgentApp
//
//  Created by caoshiqian on 2025/7/4.
//

#import "AIChatInputBar.h"

#define W ([UIScreen mainScreen].bounds.size.width)
#define H ([UIScreen mainScreen].bounds.size.height)

#define kButtonSize 40.0
#define kPanelButtonSize 60.0
#define kHorizontalPadding 5.0
#define kVerticalPadding 5.0

@interface AIChatInputBar () <UITextFieldDelegate>

@property (nonatomic, strong) UIButton *addButton;
@property (nonatomic, strong) UIButton *sendButton;
@property (nonatomic, strong) UIButton *quitButton;
@property (nonatomic, strong) UIButton *pauseButton;

@property (nonatomic, strong) UIView *mediaPanel;
@property (nonatomic, strong) UIButton *cameraButton;
@property (nonatomic, strong) UIButton *albumButton;
@property (nonatomic, assign) BOOL isMediaPanelVisible;
@property (nonatomic, assign) BOOL isKeyboardShow;

@end

@implementation AIChatInputBar

- (instancetype)initWithFrame:(CGRect)frame {
    if (self = [super initWithFrame:frame]) {
        [self setupSubviews];
        [self setupNotifications];
    }
    return self;
}

- (void)setupSubviews {
    self.backgroundColor = [UIColor colorWithWhite:0.9 alpha:1.0];
    
    self.addButton = [self buttonWithTag:BUTTON_TAG_ADD imageName:@"btn_add.png" frame:CGRectZero];
    self.pauseButton = [self buttonWithTag:BUTTON_TAG_PAUSE imageName:@"btn_pause.png" frame:CGRectZero];
    self.quitButton = [self buttonWithTag:BUTTON_TAG_QUIT imageName:@"cancel_call.png" frame:CGRectZero];
    
    [self addSubview:self.addButton];
    [self addSubview:self.pauseButton];
    [self addSubview:self.quitButton];
    
    // 输入框
    _textField = [[UITextField alloc] initWithFrame:CGRectZero];
    _textField.borderStyle = UITextBorderStyleRoundedRect;
    _textField.layer.borderColor = [UIColor blackColor].CGColor;
    _textField.backgroundColor = [UIColor whiteColor];
    _textField.textColor = [UIColor blackColor];
    _textField.returnKeyType = UIReturnKeySend;
    _textField.delegate = self;
    [self addSubview:_textField];
    
    [self setupMediaPanel];
}

- (void)setupMediaPanel {
    self.mediaPanel = [[UIView alloc] initWithFrame:CGRectZero];
    self.mediaPanel.backgroundColor = [UIColor colorWithWhite:0.9 alpha:1.0];
    self.mediaPanel.hidden = YES;
    [self addSubview:self.mediaPanel];
    
    self.cameraButton = [self buttonWithTag:BUTTON_TAG_CAMERA imageName:@"btn_camera.png" frame:CGRectZero];
    [self configPanelButton:self.cameraButton];
    [self.mediaPanel addSubview:self.cameraButton];
    
    self.albumButton = [self buttonWithTag:BUTTON_TAG_PHOTO imageName:@"btn_album.png" frame:CGRectZero];
    [self configPanelButton:self.albumButton];
    [self.mediaPanel addSubview:self.albumButton];
}

- (void)layoutSubviews {
    [super layoutSubviews];
    
    self.addButton.frame = CGRectMake(kHorizontalPadding, (kInputBarHeight - kButtonSize) / 2, kButtonSize, kButtonSize);
    self.pauseButton.frame = CGRectMake(CGRectGetWidth(self.bounds) - kButtonSize - kHorizontalPadding,
                                        (kInputBarHeight - kButtonSize) / 2, kButtonSize, kButtonSize);
    self.quitButton.frame = CGRectMake(CGRectGetMaxX(self.pauseButton.frame) - kButtonSize - kHorizontalPadding - kButtonSize,
                                       (kInputBarHeight - kButtonSize) / 2, kButtonSize, kButtonSize);
    self.textField.frame = CGRectMake(CGRectGetMaxX(self.addButton.frame) + kHorizontalPadding,
                                      (kInputBarHeight - kButtonSize) / 2,
                                      CGRectGetMinX(self.quitButton.frame) - CGRectGetMaxX(self.addButton.frame) - 2 * kHorizontalPadding,
                                      kButtonSize);
    
    // 更新媒体面板宽度
    self.mediaPanel.frame = CGRectMake(0, kInputBarHeight, CGRectGetWidth(self.bounds), kMediaPanelHeight);
    CGFloat panelW = CGRectGetWidth(self.mediaPanel.bounds);
    CGFloat margin = (panelW / 4 - kPanelButtonSize) / 2;
    self.cameraButton.frame = CGRectMake(margin, (kMediaPanelHeight - kPanelButtonSize) / 2,
                                         kPanelButtonSize, kPanelButtonSize);
    self.albumButton.frame = CGRectMake(margin + panelW / 4,
                                        (kMediaPanelHeight - kPanelButtonSize) / 2,
                                        kPanelButtonSize, kPanelButtonSize);
}

- (void)setupNotifications {
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(keyboardWillShow:)
                                                 name:UIKeyboardWillShowNotification
                                               object:nil];
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(keyboardWillHide:)
                                                 name:UIKeyboardWillHideNotification
                                               object:nil];
}

- (void)dealloc {
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}

#pragma mark - Actions

- (void)buttonAction:(UIButton *)sender {
    // 弹出媒体面板
    if (sender.tag == BUTTON_TAG_ADD) {
        [self addButtonTapped];
        return;
    }
    
    NSLog(@"button action: %@", @(sender.tag));
    if ([self.delegate respondsToSelector:@selector(inputBarActionWithTag:)]) {
        [self.delegate inputBarActionWithTag:sender.tag];
    }
}

- (void)addButtonTapped {
    if (self.isMediaPanelVisible) {
        [self hideMediaPanel];
    } else {
        [self showMediaPanel];
    }
}

#pragma mark - UI

- (BOOL)textFieldShouldReturn:(UITextField *)textField {
    if (textField.text.length > 0 && [self.delegate respondsToSelector:@selector(inputBarDidSendText:)]) {
        [self.delegate inputBarDidSendText:textField.text];
        textField.text = @"";
    }
    return YES;
}

- (UIButton *)buttonWithTag:(NSInteger)tag imageName:(NSString *)imageName frame:(CGRect)frame {
    UIButton *btn = [UIButton buttonWithType:UIButtonTypeCustom];
    btn.frame = frame;
    [btn setImage:[UIImage imageNamed:imageName] forState:UIControlStateNormal];
    [btn addTarget:self action:@selector(buttonAction:) forControlEvents:UIControlEventTouchUpInside];
    btn.tag = tag;
    return btn;
}

- (void)configPanelButton:(UIButton *)button {
    button.backgroundColor = [UIColor colorWithWhite:0.7 alpha:1.0];
    button.layer.cornerRadius = 10;
    button.layer.masksToBounds = YES;
    button.imageEdgeInsets = UIEdgeInsetsMake(10, 10, 10, 10);
}

#pragma mark - Animation

- (void)showMediaPanel {
    if (self.isMediaPanelVisible) return;
    
    self.mediaPanel.hidden = NO;
    [UIView animateWithDuration:0.25 animations:^{
        CGRect frame = self.frame;
        frame.size.height = [self barHeight:YES isBottom:!self.isKeyboardShow];
        frame.origin.y = self.superview.bounds.size.height - frame.size.height;
        self.frame = frame;
        
        self.mediaPanel.frame = CGRectMake(0, kInputBarHeight, CGRectGetWidth(self.bounds), kMediaPanelHeight);
    }];
    
    _isMediaPanelVisible = YES;
}

- (void)hideMediaPanel {
    if (!self.isMediaPanelVisible) return;
    
    [UIView animateWithDuration:0.25 animations:^{
        CGRect frame = self.frame;
        frame.size.height = [self barHeight:NO isBottom:!self.isKeyboardShow];
        frame.origin.y = self.superview.bounds.size.height - frame.size.height;
        self.frame = frame;
    } completion:^(BOOL finished) {
        self.mediaPanel.hidden = YES;
    }];
    
    _isMediaPanelVisible = NO;
}

- (void)keyboardWillShow:(NSNotification *)notification {
    self.isKeyboardShow = YES;
    self.isMediaPanelVisible = NO;
    self.mediaPanel.hidden = YES;
}

- (void)keyboardWillHide:(NSNotification *)notification {
    self.isKeyboardShow = NO;
    self.isMediaPanelVisible = NO;
    self.mediaPanel.hidden = YES;
}

- (CGFloat)safeAreaBottom {
    CGFloat safeAreaBottom = 0.f;
    if (@available(iOS 11.0, *)) {
        safeAreaBottom = kSafeAreaBottom;
    }
    return safeAreaBottom;
}

- (CGFloat)barHeight:(BOOL)panelVisible isBottom:(BOOL)bottom {
    CGFloat height = panelVisible ? kInputBarHeight + kMediaPanelHeight : kInputBarHeight;
    if (bottom) {
        height += [self safeAreaBottom];
    }
    return height;
}

@end
