//
//  BaiduRtcRoomApi.h
//  rtc_sdk_ios
//
//  Created by Sun,Jian(ACU-T102) on 2018/12/5.
//  Copyright © 2018年 Sun,Jian(ACU-T102). All rights reserved.
//

#ifndef BaiduRtcRoomApi_h
#define BaiduRtcRoomApi_h
#import <BaiduRtcRoom/BaiduRtcRoomApiDefines.h>
#import <BaiduRtcRoom/BaiduRtcRoomApiAudioRecord.h>
#import <BaiduRtcRoom/BaiduRtcRoomApiAudioSession.h>
#import <BaiduRtcRoom/BaiduRtcRoomApiVideoExternalRender.h>
#import <BaiduRtcRoom/BaiduRtcRoomApiAudioExternalDevice.h>
#import <BaiduRtcRoom/BaiduRtcExtension.h>
#import <BaiduRtcRoom/BaiduRtcVideoProcessor.h>
#import <BaiduRtcRoom/RTCMediaCaptureDefine.h>
#import <BaiduRtcRoom/BRTCMediaRecorder.h>
#import <BaiduRtcRoom/RTCLocalVideoView.h>
#import <BaiduRtcRoom/RTCRemoteVideoView.h>
#import <BaiduRtcRoom/BaiduRtcRoomDelegate.h>


@interface BaiduRtcRoomApi : NSObject
/**
 BaiduRtcRoom SDK 版本号
 
 @return 版本号
 @discussion 开发者使用本 API 获取 SDK 版本号
 */
+ (NSString *)version;


/**
 是否打开调试信息
 
 @param bOnVerbose 是否打开调试信息，true 打开，false 不打开。默认为 false
 @discussion 建议在初始化 SDK 前调用。建议在调试阶段打开此开关，方便调试
 */
+ (void)setVerbose:(BOOL)bOnVerbose;

/**
 * called by user for AudioUnitDevice reboot
 */
- (BOOL)resetAudioDeviceConfigure;

/**
 初始化 SDK
 
 @param appId  百度 派发的数字 ID, 开发者的唯一标识
 @param tokenStr  app server 派发的token字符串, 用来校验对应signal 通信的合法性
 @return SDK 实例，nil 表示初始化失败
 @discussion 初始化 SDK 时调用。初始化 SDK 失败可能导致 SDK 功能异常
 */
- (instancetype)initSDKWithAppID:(NSString *)appId
                        tokenStr:(NSString *)tokenStr
                        delegate:(id<BaiduRtcRoomDelegate>)delegate;


#pragma mark -- av parameter setting
/**
 设置音视频相关的参数

 @param paramSettings 该参数封装了音视频的一些参数,如video分辨率，fps, bitrate，音频采样率等
 @param paramType 参数类型，可指定设置某一项，还是所有参数都设置
 @discussion 该函数在loginRoom 前调用，主要用于设置音视频采集，编解码相关的参数
 */
- (void)setParamSettings:(RtcParameterSettings *)paramSettings paramType:(RtcParamSettingType)paramType;

/**
 获取音视频相关的参数
 
 @discussion 获取当前音视频采集，编解码等相关的参数
 */
- (RtcParameterSettings *)getParamSettings;

/**
 * 设置音频配置 profile
 * @param profile 音频采样率、音频通道、音频编码、码率配置, 详见 {@link  BRTCAudioProfileType};
 * @param scenario 音频应用场景设置, 详见 {@link BRTCAudioScenario}
 * @apiNote 该方法需要在loginRtcRoomWithRoomName之前调用，之后设置则不生效；可以配合setParamSettings 微调各个具体音频参数
 * @return 接口调用成功返回0
 */
- (int) setAudioProfile:(BRTCAudioProfileType) profile  scenrio:(BRTCAudioScenario) scenario;

/**
 RTC引擎统计信息
 
 @param bOnStatistics 是否打开rtc引擎统计信息， bOnStatistics = YES,打开开关，bOnStatistics = NO, 关闭引擎统计信息
 @discussion 引擎统计信息开关,当打开开关时 onEngineStatisticsInfo 函数会每隔2秒返回引擎的统计信息，并且可通过queryEngineStatisticsInfo主动函数查询到引擎统计信息
 */
- (void)setEngineStateStatistics:(BOOL)bOnStatistics;

/**
 RTC质量监控数据上报
 
 @param isEnable 是否打开rtc质量监控数据上报   isEnable = YES, 开启上报，isEnable = NO, 关闭上报
 @param qualityMonitorEnv   线上环境："online"   沙盒："qa"，用于测试
 @discussion 预置接口 监控信息上报开关   当打开开关时，上报帧率、码率、分辨率、丢包率等监控信息到服务端，console可查
 */
- (void)enableStatsToServer:(BOOL)isEnable qualityMonitorEnv:(NSString *)qualityMonitorEnv;

/**
 RTC异常信息上报
 
 @param enableErrorInfoReport 是否打开上报开关 YES, 开启上报 NO, 关闭上报。默认公网域名开启上报
 @discussion 上报异常信息，方便问题排查
 */
- (void)enableErrorInfoReprot:(BOOL)enableErrorInfoReport;


/**
 * RTC 发送SEI消息
 * @param message 二进制消息
 * @param repeatCount 重复次数
 */
- (void)sendSEIMsg:(NSData *)message repeatCount:(int)repeatCount;



/**
 * 角色转换
 * @param roleType 角色类型
 */
- (void)switchRole:(BRTCRoleType)roleType;

#pragma mark -- 登录／退出房间
/**
 登录房间
 
 @param roomName 房间名，长度不可超过 255 byte
 @param userId 用户id,每个房间的用户ID必须唯一
 @param displayName 用户昵称
 @return true 成功，false 失败
 @discussion 登录房间成功，在同一个房间的人能进行相互音视频聊天，如果失败，会通过onErrorInfoUpdate call back 返回错误信息
 */
- (BOOL)loginRtcRoomWithRoomName:(NSString *)roomName
                          userID:(NSInteger)userId
                     displayName:(NSString *)displayName;

/**
 登录房间
 
 @param roomName 房间名，长度不可超过 255 byte
 @param userId 用户id,每个房间的用户ID必须唯一
 @param displayName 用户昵称
 @param options 其他登录参数
 @return true 成功，false 失败
 @discussion 登录房间成功，在同一个房间的人能进行相互音视频聊天，如果失败，会通过onErrorInfoUpdate call back 返回错误信息。建议在断网重连或者初次登录失败时调用
 */
- (BOOL)loginRtcRoomWithRoomName:(NSString *)roomName
                          userID:(NSInteger)userId
                     displayName:(NSString *)displayName
                         options:(LoginOptions *)options;

/**
 退出房间
 
 @return true 成功，false 失败
 @discussion 执行logoutRtcRoom后，会停止音视频采集，断开与房间服务器的连接，取消音视频的传输，销毁音视频传输通道以及释放其他资源。
 */
- (BOOL)logoutRtcRoom;



#pragma mark -- 发布／订阅流接口

/// 发布流
/// @discussion 流发布在roomId指定的房间，在同一房间joined 的用户可以相互订阅流，默认在发布流的同时，listening/subscriber 在该房间其他用户的流；
- (void)publishStreaming;

/**
 停止发布流
 @param isLeaving 是否退出房间并销毁相关资源
 @discussion stop 通过 publishStreaming 发布的流
 */
- (void)stopPublishStreaming:(BOOL)isLeaving;

/// 开始发布视频流, 仅支持重构版本
- (void)publishVideoStreaming;

/// 停止发布视频流, 仅支持重构版本
- (void)unpublishVideoStreaming;

/// 订阅流
/// @param streamingIds 用户要订阅的其他用户的流id列表(即其他用户id列表)
/// @discussion 用于订阅同一房间的其他用户的流
- (void)subsribeStreaming:(NSArray<NSNumber *> *)streamingIds;

/*
 * 订阅流，支持本地方位混音
 *
 * @param streamingIds, 用户要订阅的其他用户的流id列表(即其他用户id列表)
 * @param horiAngle 相对水平角
 * @param vertAngle 相对俯仰角
 * @param distance   距离
 * @param nearVol   本地音量
 * @param mixerVol  背景音
 *
 * @discuss 用于订阅同一房间的其他用户的流
 *
 **/
- (void)subsribeStreaming:(NSArray<NSNumber *> *)streamingIds
               horiAngle:(float)horiAngle
                vertAngle:(float)vertAngle
                 distance:(float)distance
                  nearVol:(int)nearVol
                 mixerVol:(int)mixerVol;

/*
 * 订阅流，更新本地方位混音参数
 * 建议配合(订阅流，支持本地方位混音)使用
 * @param streamingIds, 用户要订阅的其他用户的流id列表(即其他用户id列表)
 * @param horiAngle 相对水平角
 * @param vertAngle 相对俯仰角
 * @param distance   距离
 * @param nearVol   本地音量
 * @param mixerVol  背景音
 *
 * @discuss 用于订阅同一房间的其他用户的流
 *
 **/
- (void)updateSubsribeStreaming:(NSArray<NSNumber *> *)streamingIds
                      horiAngle:(float)horiAngle
                      vertAngle:(float)vertAngle
                       distance:(float)distance
                        nearVol:(int)nearVol
                       mixerVol:(int)mixerVol;


/*
 * 更新推流者位置信息
 * @param uid 用户uid
 * @param posX 坐标x
 * @param posY 坐标y
 * @param posZ 坐标z
 */
- (void)publisherStreamingPosition:(NSInteger)uid
                              posX:(int)posX
                              posY:(int)posY
                              posZ:(int)posZ;

/// 订阅指定远端用户音频流
/// @param feedId 远端用户 ID
- (void)subscribeAudioStreaming:(NSNumber *)feedId;

/// 订阅指定远端用户视频流
/// @param feedId 远端用户 ID
- (void)subscribeVideoStreaming:(NSNumber *)feedId;

/// 订阅所有指定远端用户音频流
- (void)subscribeAllRemoteAudioStreams;

/// 订阅所有指定远端用户视频流
- (void)subscribeAllRemoteVideoStreams;

/// 停止订阅流
/// @param streamingIds 用户要停止订阅的流id列表(即其他用户id列表)
- (void)stopSubscribeStreaming:(NSArray<NSNumber *> *)streamingIds;

/// 停止订阅指定远端用户音频流
/// @param feedId 远端用户 ID
- (void)stopSubscribeAudioStreaming:(NSNumber *)feedId;

/// 停止订阅指定远端用户视频流
/// @param feedId 远端用户 ID
- (void)stopSubscribeVideoStreaming:(NSNumber *)feedId;

/// 停止订阅所有指定远端用户音频流
- (void)stopSubscribeAllRemoteAudioStreams;

/// 停止订阅所有指定远端用户视频流
- (void)stopSubscribeAllRemoteVideoStreams;

#pragma mark - 屏幕分享

/// 开始应用内屏幕分享
/// @discussion 仅抓取宿主应用的屏幕内容
- (void)startShareAppScreen API_AVAILABLE(ios(11.0));
 
/// 开始系统屏幕分享
/// @param appGroup 用于宿主应用与录屏进程共享的 Application Group Identifier
/// @discussion 支持抓取整个 iOS 系统的屏幕
- (void)startShareSystemScreenWithAppGroup:(NSString *)appGroup API_AVAILABLE(ios(11.0));
 
/// 停止屏幕分享
- (void)stopShareScreen;

/// 更新屏幕分享配置参数
/// @param params 屏幕流音视频参数
/// @discussion 该接口须在 startShareAppScreen 或 startShareSystemScreenWithAppGroup: 之后调用
/// @return 参数更新结果
- (int)updateScreenShareParams:(BRTCScreenShareParams *)params;

#pragma mark -- custom commands
/**
 踢出某人
 
 @param userId 在房间中的用户的 用户ID
 @discussion 房管／主播／会议主持 把某人踢出聊天室
 */
- (void)kickOffUserWithId:(NSInteger)userId; //add roomID


/**
 禁言某人
 
 @param userId 在房间中的用户的 用户ID
 @param isDisable 是否禁言某人
 @discussion  房管/主播／会议主持 禁止某人发言
 */
- (void)shutUpUserWithId:(NSInteger)userId isDisable:(BOOL)isDisable;

/**
  解散房间
 
  @discussion 解散房间，房间管理员有权利解散整个房间，解散后，房间中的每个人都退出房间
 */
- (void)disbandRoom;

#pragma mark - Beauty
#if TARGET_OS_IPHONE
/// 获取 BRTC 内部美颜管理类，用于开关及控制 BRTC 自带的相机流美颜效果
- (id<BaiduRtcBeautyManager>)getRtcBeautyManager;

//获取 BRTC 的视频管理类，以处理水印功能
- (id<BaiduRtcVideoProcessManager>)getRtcVideoProcessManager;

//获取 BRTC 录制对象
- (BRTCMediaRecorder *)getRtcMediaRecorder;
#endif
#pragma mark - Video Process

/// 设置自定义相机流编码前处理节点
/// @discussion 若使用 BRTC 自带的相机采集器，则可以通过该方法进行自定义前处理行为，
/// 若已经通过 setMediaCapturer: 设置了 自定义采集器，既可以通过该方法进行自定义前处理，
/// 也可以在 自定义采集器 内部执行完前处理，再将数据输出到回调
- (void)setVideoProcessor:(BaiduRtcVideoProcessor *)processor;

#pragma mark - Audio Process

/// 获取 BRTC 内部音频处理管理类，用于控制 变声音效、背景音效 等音频处理效果
- (id<BaiduRtcAudioProcessManager>)getRtcAudioProcessManager;

#pragma mark -- video devices setting

/// 设置自定义主视频源
/// @param capturer 视频数据采集器
/// @discussion 主视频流一般为相机流，当接入方通过调用该方法设置自定义源，内部相机采集将不会启动
- (void)setVideoCapturer:(id<BaiduVideoCapturer>)capturer;

/// 为指定类型的流设置自定义视频数据采集器
/// @param capturer 媒体数据采集器
/// @param mediaTarget 媒体流类型
- (void)setMediaCapturer:(id<BaiduMediaCapturer>)capturer forMediaTarget:(RtcMediaTarget)mediaTarget;

/**
 设置外部渲染回调对象
  
 @param renderDelegate 遵循 BaiduRtcApiRenderDelegate 协议的代理对象
 @discussion 使用外部渲染功能，需要设置代理对象。未设置代理对象，或对象设置错误，可能导致无法正常收到相关回调
 */
- (void)setRenderDelegate:(id<BaiduRtcApiRenderDelegate>)renderDelegate;


/**
 本地预览
 
 @discussion 打开camera，开始预览
 */
- (void)startPreview;


/**
 停止预览
 
 @discussion 关闭camera, 停止本地预览
 */
- (void)stopPreview;



/**
 切换摄像头
 
 @discussion 前后摄像头切换
 */
- (void)switchCamera;

/**
 可以暂停（或恢复）发布本地的视频画面，暂停之后，同一房间中的其他用户将无法继续看到自己画面
 等效于 startPreview / stopPreview 这两个接口，但具有更好的响应速度
 @param isMute 设置指定的状态摄像头状态 YES mute
 */
- (void)muteCamera:(BOOL)enabled;

/**
 设置前后摄像头
 
 @param front true:前置 false:后置
 @discussion 设置前后摄像头
 */
-(void)setCameraFace:(BOOL)front;

/**
 摄像头对焦
 
 @param point 对焦点坐标
 @param size camera预览view尺寸
 @discussion 摄像头对焦功能，支持手动和自动对焦
 */
- (void)cameraFocusWithPoint:(CGPoint)point andPlaneSize:(CGSize)size;
#if TARGET_OS_IPHONE
/// 是否允许根据设备方向自动更新采集输出方向，默认为 YES
/// @param enabled enabled
- (void)enableCameraAutoUpdateOrientation:(BOOL)enabled;

/// 设置相机采集默认输出方向，默认为 UIDeviceOrientationPortrait
/// @param orientation 方向
- (void)setCameraOutputOrientation:(UIDeviceOrientation)orientation;
#endif
#pragma mark - 音频内部设备设置

/**
 设置音频对外暴露回调
 
 @param delegate  遵循 BaiduRtcApiAudioFrameDelegate 协议的代理对象
 @discussion 开启音频录制功能，需要设置代理对象。未设置代理对象，或对象设置错误，可能导致无法正常收到相关回调
 */
- (void)setAudioFrameDelegate:(id<BaiduRtcApiAudioFrameDelegate>)delegate;

/**
 设置音频会话代理对象

 @param audioSessionDelegate 遵循 BaiduRtcRoomApiAudioSessionDelegate 协议的代理对象
 @discussion 接收音频设备状态变化的通知，详情请看BaiduRtcRoomApiAuidoSession.h
 */
- (void)setAudioSessionDelegate:(id<BaiduRtcRoomApiAudioSessionDelegate>)audioSessionDelegate;

#pragma mark - 音频外部设备设置
/**
 设置音频外部采集代理对象

 @param audioExternalDelegate 遵循 BaiduRtcRoomApiAudioExternalDeviceDelegate 协议的代理对象
 @discussion 接收音频外部采集的回调功能。
 */
- (void)setAudioExternalDeviceDelegate:(id<BaiduRtcRoomApiAudioExternalDeviceDelegate>)audioExternalDelegate;

/// 设置自定义采集音频音量
/// @param volume 音量。范围[0, 4].
- (void)setLocalAudioExternalCaptureVolume:(float)volume;

#pragma mark - 音频设备控制

/**
 关闭/开启本地麦克风
 @param enabled  disabled local mic capture NO; enable/restart local mic capture YES，return 0 success, otherwish -1 failure
 @discussion 关闭麦克风，停止音频的采集
 */
- (int)enableLocalAudio:(BOOL)enabled;

/**
 关闭扬声器
 
 @discussion 关闭扬声器，停止音频播放
 */
- (void)muteSpeaker:(BOOL) isMute;

/**
 打开/关闭麦克风
 
 @param isMute true:关闭 false:开启 默认开启
 @discussion 关闭麦克风，停止音频的采集
 */
-(void)muteMicphone:(BOOL)isMute;

/**
 开关扬声器
 
 @discussion 听筒与扬声器切换(免提功能)
 */
- (void)switchLoundSpeaker;

/**扬声器和听筒切换
 */
- (void)switchAudioCategaryWithSpeaker:(BOOL)isSpeaker;

/**
 预置听筒/扬声器
 
 @param isPresetLoudSpeaker   true:扬声器   false:听筒    默认听筒
 @discussion 在initSDKWithAppID之后，loginRoom之前调用，预置听筒/扬声器播放语音
 */
-(void)presetLoudSpeaker:(BOOL)isPresetLoudSpeaker;

/// 音频路由是否是系统扬声器
/// @return 系统扬声器状态，YES 表示当前通话音频从系统扬声器输出，NO 表示从其他路由输出,如听筒、耳机、蓝牙设备等
- (BOOL)isSpeakerOn;

#pragma mark - 音频信号控制

/**
 语音激励接口
 
 @return NSDictionary* 用户音量大小
 @discussion 获取语音激励
 */
-(NSArray<RtcRoomAudioLevel *> *)getRemoteAudioLevels;

/// 启停用户音量提示
/// @param enable 控制启用或停止音量提示
/// @param interval 提示回调间隔，单位毫秒。若启用，不能低于 100ms，建议 300ms
/// @apiNote 该方法需要在loginRtcRoomWithRoomName之后调用，之前设置不生效
- (void)enableAudioVolumeIndication:(BOOL)enable interval:(int)interval;

/**
 设置音频 session mode
 
 @param audioSessionMode 音频mode
 @discussion 设置音频 session mode
 */
- (void)setAudioSessionMode:(AVAudioSessionMode)audioSessionMode;

/**
 是否开启音频自动增益
 
 @param enable true:开启 false:关闭  默认false
 @discussion  是否开启音频自动增益
 */
- (void)enableAgc:(BOOL)enable;

/**
 是否开启音频噪声抑制
 
 @param enable true:开启 false:关闭  默认false
 @discussion   是否开启音频噪声抑制
 */
- (void)enableAns:(BOOL)enable;

/**
 设置声音输出方式
 
 @param mode 可选值：speaker（扬声器）或 ear（听筒)
 @discussion  设置声音输出方式
 */
- (void)setSoundMode:(RtcSoundMode)mode;

/// 设置本地采集音频音量
/// @param volume 音量。范围[0, 4].
- (void)setLocalAudioCaptureVolume:(float)volume;

/**
 指定远端画面暂停/恢复播放
 
 @ stats true播放  false暂停。
 @param userId 用户userID
 @discussion  控制指定远端用户画面暂停/恢复播放. 该设置对房间内所有订阅者全部生效。
 */
-(void)setRemoteAudioPlayState:(BOOL)stats userId:(NSInteger)userId;

/**
 指定用户音量.
 
 @param userId 用户userID
 @param volume 音量。范围[0,1].
 @discussion  指定用户音量. 音量设置只在本端有效，其他订阅者无影响
 */
- (void)setRemoteAudioPlayVolume:(float)volume userId:(NSInteger)userId;

/// 设置所有远端用户混音后音量
/// @param volume 音量。范围[0, 4].
- (void)setRemoteAudioPlayVolume:(float)volume;

#pragma mark -- video display
/**
 设置本地显示view
 
 @param localVideoView 本地显示view,用于显示camera采集的视频数据
 @discussion 在loginRoom之前调用，loginRoom之后，本地视频数据会显示到 localVideoView,
             localVideoView的位置大小要与采集的视频大小成比例。
 */
//#if TARGET_OS_IPHONE
- (void)setLocalDisplay:(RTCLocalVideoView *)localVideoView;
//#elif TARGET_OS_MAC
//- (void)setLocalDisplay:(NSView<RTCVideoRenderer> *)localVideoView;
//#endif

/**
 设置本地关联指定媒体目标类型的显示view
 
 @param localVideoView 本地显示view,用于显示camera采集的视频数据
 @param mediaTarget 视图需要绑定到的媒体目标
 @discussion 在loginRoom之前调用，loginRoom之后，本地视频数据会显示到 localVideoView,
             localVideoView的位置大小要与采集的视频大小成比例。
 */
//#if TARGET_OS_IPHONE
- (void)setLocalDisplay:(RTCLocalVideoView *)localVideoView mediaTarget:(RtcMediaTarget)mediaTarget;
//#elif TARGET_OS_MAC
//- (void)setLocalDisplay:(NSView<RTCVideoRenderer> *)localVideoView mediaTarget:(RtcMediaTarget)mediaTarget;
//#endif

/**
 设置指定用户的远端显示view
 
 @param remoteVideoView 远端画面显示view, 用于显示远端用户传输过来的视频数据
 @param userId 用户userID
 @discussion 在远端用户流到达后调用，指定用户视频数据会显示到 remoteVideoView，
 remoteVideoView 的大小要与采集的视频大小成比例
 */
//#if TARGET_OS_IPHONE
- (void)setRemoteDisplay:(RTCRemoteVideoView *)remoteVideoView userId:(NSInteger)userId;
//#elif TARGET_OS_MAC
//- (void)setRemoteDisplay:(NSView<RTCVideoRenderer> *)remoteVideoView userId:(NSInteger)userId;
//#endif

/**
 设置远端指定用户关联指定媒体目标类型的显示 view.
 
 
 
 @param remoteVideoView 远端画面显示view, 用于显示远端用户传输过来的视频数据
 @param userId 用户userID
 @param mediaTarget 视图需要绑定到的媒体目标
 @discussion 在远端用户流到达后调用，指定用户视频数据会显示到 remoteVideoView，
 remoteVideoView 的大小要与采集的视频大小成比例
 */
//#if TARGET_OS_IPHONE
- (void)setRemoteDisplay:(RTCRemoteVideoView *)remoteVideoView userId:(NSInteger)userId mediaTarget:(RtcMediaTarget)mediaTarget;
//#elif TARGET_OS_MAC
//- (void)setRemoteDisplay:(NSView<RTCVideoRenderer> *)remoteVideoView userId:(NSInteger)userId mediaTarget:(RtcMediaTarget)mediaTarget;
//#endif

/**
 更新指定用户的远端显示view
 
 @param remoteVideoView 远端画面显示view, 用于显示远端用户传输过来的视频数据
 @param userId 用户userID
 @discussion 更新指定用户视频数据显示view
 remoteVideoView 的大小要与采集的视频大小成比例
 */
//#if TARGET_OS_IPHONE
- (void)updateDisplay:(RTCRemoteVideoView *)remoteVideoView userId:(NSInteger)userId;
//#elif TARGET_OS_MAC
//- (void)updateDisplay:(NSView<RTCVideoRenderer> *)remoteVideoView userId:(NSInteger)userId;
//#endif



/**
 指定远端画面暂停/恢复播放
 
 @param stats true播放 false暂停
 @param userId 用户userID
 @discussion  控制指定远端用户画面暂停/恢复播放
 */
-(void)setRemoteVideoPlayState:(BOOL)stats userId:(NSInteger)userId;
#if TARGET_OS_IPHONE
/// 视频画面截图
/// @param userId 视频流用户 ID，0 表示截取本地视频画面
/// @param mediaTarget 流类型，如果传入 nil 则指定为默认视频流
/// @param completionBlock 截图完成回调
/// @discussion 对本地用户仅支持默认视频流
- (void)takeVideoSnapshot:(NSInteger)userId mediaTarget:(RtcMediaTarget)mediaTarget completionBlock:(void (^)(UIImage *image, RtcErrorCodes errCode))completionBlock;
#elif TARGET_OS_OSX
- (void)takeVideoSnapshot:(NSInteger)userId mediaTarget:(RtcMediaTarget)mediaTarget completionBlock:(void (^)(NSImage *image, RtcErrorCodes errCode))completionBlock;
#endif
#pragma mark -- beauty interface
/**
 设置美白度
 
 @param factor 在美颜中的美白因子
 @discussion 美白系数调节
 */
- (void)setWhitenFactor:(float)factor;



/**
 设置红润度
 
 @param factor 在美颜中的红润因子
 @discussion 红润系数调节
 */
- (void)setRedenFactor:(float)factor;



/**
 设置磨皮度
 
 @param factor 在美颜中的磨皮因子
 @discussion 磨皮系数调节
 */
- (void)setBuffingFactor:(float)factor;


/**
 查询房间用户信息
 
 @return NSArray* Message用户信息列表
 @discussion 获取房间中Message用户的列表
 */
- (NSArray *)queryMessageUserListOfRoom;


/**
 查询RTC引擎统计信息
 
 @return NSArray* rtc engine状态统计信息
 @discussion 用于获取rtc engine 的一些运行状态信息，如cpu, video codec, fps等信息
 */
- (NSArray *)queryEngineStatisticsInfo;


#pragma mark -- sen text message in the data channel
/**
 发送文本消息

 @param message 需要发送的文本消息
 @return int, -1 发送失败，0 发送成功
 @discussion 当enable数据通道后，可通过该接口发送文本消息给在同一房间的其他用户
 */
- (int)sendMessage:(NSString *)message;

/**
 发送文本消息
 
 @param message 需要发送的文本消息
 @param userId 接收信息的用户ID
 @return int, -1 发送失败，0 发送成功
 @discussion 当enable数据通道后，该接口发送文本消息给在同一房间的指定用户
 */
- (int)sendMessage:(NSString *)message withUserId:(NSNumber *)userId;

/**
 发送文本消息
 
 @param message 需要发送的文本消息
 @return int, -1 发送失败，0 发送成功
 @discussion 通过信令信道 该接口发送文本消息给在同一房间的其他用户
 */
- (int)sendMessage2:(NSString *)message;

/**
 发送文本消息
 
 @param message 需要发送的文本消息
 @param userId 接收信息的用户ID
 @return int, -1 发送失败，0 发送成功
 @discussion 通过信令信道 该接口发送文本消息给在同一房间的指定用户
 */
- (int)sendMessage2WithUserId:(NSString *)message
                  userId:(NSNumber *)userId;

/**
 设置用户attribute属性
 
 @param attribute attribute值
 @return int, -1 发送失败，0 发送成功
 @discussion  设置本用户attribute属性
 */
- (int)setUserAttribute:(NSString *)attribute;

/**
 获取用户attribute属性
 
 @param userID 用户UID
 @return int, -1 发送失败，0 发送成功
 @discussion  获取指定用户attribute属性，属性值由onTextMessageAttribute回调返回
 */
- (int)getUserAttribute:(NSNumber *)userID;


#pragma mark -- configure live server
/**
 配置server端推流参数

 @param url rtmp推流地址
 @param isMix 是否做混流处理
 @param isRecording 是否录制
 @param mixTmplt 混流模版
 @param mode 转推模式，有两种： 聊天室模式， 主播转推模式
 @param avParam 转推音视频参数
 @discussion 该接口用于配置server推流的参数，聊天室模式：在同一个rtc房间的所有参与者在混流后，直接转推到一个指定的直播房间；
             主播转推模式：主播推向不同的直播房间。
 */
- (void)configLiveServerWithUrl:(NSString *)url
                          isMix:(BOOL)isMix
                        isRecording:(BOOL)isRecording
                    mixTemplate:(NSString *)mixTmplt
                   transferMode:(RtcLiveTransferMode)mode
                        avParam:(RtcLiveAuidoVideoParameters*)avParam;

/**
 动态配置server端推流参数
 
 @param url rtmp推流地址
 @param isMix 是否做混流处理
 @param isRecording 是否录制
 @param mixTmplt 混流模版
 @param mode 转推模式，有两种： 聊天室模式， 主播转推模式
 @discussion 该接口用于动态配置server推流的参数。configLiveServerWithUrl在执行loginRtcRoom之前调用。本接口在loginRtcRoom成功后调用。两个接口不要混用
 */
- (void)startLiveServerStreaming:(NSString *)url
                          isMix:(BOOL)isMix
                    isRecording:(BOOL)isRecording
                    mixTemplate:(NSString *)mixTmplt
                    transferMode:(RtcLiveTransferMode)mode;

/**
 停止server转推
 
 @param transferMode 转推模式，有两种： 聊天室模式， 主播转推模式
 @discussion 若需要通话过程中，停止转推，调用该接口
 */
- (void)stopLiveServerStreaming:(RtcLiveTransferMode)transferMode;

/**
 设置本流在混流模版布局中的位置信息
 
 @param idx   混流模版中的位置索引名
 @discussion  设置本流在混流模版布局中的位置信息
 */
- (BOOL)setMixLayoutPositionIndex:(NSString *)idx;

/**
 启动跨房间连麦
 
 @param destRoomName 目标房间
 @param userId 加入目标房间时使用的userId，必须在加入房间中是唯一的
 @param token app server 派发的token字符串
 @discussion  启动跨房间连麦
 */
- (void)startRoomMediaRelay:(NSString *)destRoomName
                     userId:(NSInteger)userId
                      token:(NSString *)token;
/**
 停止跨房间连麦
 
 @param destRoomName 目标房间
 @param userId 加入目标房间时使用的userId，必须在加入房间中是唯一的
 @discussion 停止跨房间连麦
 */
- (void)stopRoomMediaRelay:(NSString *)destRoomName
                    userId:(NSInteger)userId;

/**
 停止所有跨房间连麦
 
 @discussion 停止所有启动的跨房间连麦
 */
- (void)stopRoomMediaRelayAll;
#pragma mark -- setting signal server url
/**
 设置信令server url
 @param serverUrl 信令server url
 @discussion 在loginRoom之前调用，用于配置信令server
 */
- (void)setSignalServerWithUrl:(NSString *)serverUrl;

/**
 设置janus server ip
 @param serverIp janus server ip
 @discussion 在loginRoom之前调用，设置janus server ip。
 */
- (void)setMediaServerIp:(NSString *)serverIp;

/**
  * 设置媒体信息,小度专用,格式"v=BRTC.RTOS.SDK V3.0.4B02 ASR&vc=h263&vp=176x144&ac=amr-wb"
  */
- (void)setRemoteMediainfo:(NSString *)mediainfo;

- (NSString *)getRemoteMediainfo;
@end


#endif /* BaiduRtcRoomApi_h */
