/*
 * Copyright (C) 2025 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.ai.sample.util;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import java.util.ArrayList;
import java.util.List;

import com.baidu.ai.sample.data.CharacterData;
import com.baidu.ai.sample.data.CharacterData.CustomData;
import com.baidu.ai.sample.data.CharacterData.ChineseComment;
import com.baidu.ai.sample.data.CharacterData.EnglishComment;

public class CharacterDataParser {
    /**
     * 判断字符串是否包含CUSTOM标签
     */
    public static boolean hasCustomTag(String content) {
        if (content == null || content.trim().isEmpty()) {
            return false;
        }
        String trimmed = content.trim();
        return trimmed.contains("((CUSTOM:") && trimmed.endsWith("))");
    }

    /**
     * 解析JSON字符串
     */
    public static CharacterData parseJsonString(String jsonString) {
        try {
            // 首先检查是否包含CUSTOM标签
            if (!hasCustomTag(jsonString)) {
                return null;
            }

            String jsonContent = extractJsonContent(jsonString);
            JSONObject jsonObject = new JSONObject(jsonContent);

            CharacterData characterData = new CharacterData();
            characterData.setCustom(parseCustomData(jsonObject));
            return characterData;
        } catch (JSONException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 提取JSON内容（移除外层的 (( 和 ))）
     */
    private static String extractJsonContent(String originalString) {
        String content = originalString.trim();
        if (content.startsWith("((CUSTOM:") && content.endsWith("))")) {
            content = content.substring(9, content.length() - 2);
        }
        return content;
    }

    /**
     * 解析CUSTOM数据
     */
    private static CustomData parseCustomData(JSONObject customObject) throws JSONException {
        CustomData customData = new CustomData();

        customData.setNameCh(customObject.getString("name_ch"));
        customData.setNameEn(customObject.getString("name_en"));

        // 解析中文注释数组
        JSONArray commentsChArray = customObject.getJSONArray("comments_ch");
        List<CharacterData.ChineseComment> chineseComments = new ArrayList<>();
        for (int i = 0; i < commentsChArray.length(); i++) {
            JSONObject commentObj = commentsChArray.getJSONObject(i);
            ChineseComment comment = new ChineseComment();
            comment.setName(commentObj.getString("name"));
            comment.setStrokenumber(commentObj.getInt("strokenumber"));
            comment.setRadical(commentObj.getString("radical"));
            comment.setComment(commentObj.getString("comment"));
            comment.setExampleSentence(commentObj.getString("example_sentence"));
            chineseComments.add(comment);
        }
        customData.setCommentsCh(chineseComments);

        // 解析英文注释
        JSONObject commentsEnObject = customObject.getJSONObject("comments_en");
        EnglishComment englishComment = new EnglishComment();
        englishComment.setName(commentsEnObject.getString("name"));
        englishComment.setSoundmarkUs(commentsEnObject.getString("soundmark_us"));
        englishComment.setSoundmarkEn(commentsEnObject.getString("soundmark_en"));
        englishComment.setComment(commentsEnObject.getString("comment"));
        englishComment.setExampleSentence(commentsEnObject.getString("example_sentence"));
        customData.setCommentsEn(englishComment);

        return customData;
    }

    /**
     * 获取CUSTOM标签内的纯JSON内容（如果存在）
     */
    public static String getPureJsonContent(String content) {
        if (!hasCustomTag(content)) {
            return null;
        }
        return extractJsonContent(content);
    }
}