package com.baidu.ai.sample.adapter;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.text.TextUtils;
import android.text.style.URLSpan;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.ImageView;
import android.widget.TextView;
import androidx.annotation.NonNull;
import io.noties.markwon.AbstractMarkwonPlugin;
import io.noties.markwon.Markwon;
import io.noties.markwon.MarkwonSpansFactory;
import io.noties.markwon.core.CoreProps;

import com.baidu.ai.sample.R;
import com.baidu.ai.sample.WebViewActivity;
import com.baidu.ai.sample.data.ChatBean;

import java.io.File;
import java.util.ArrayList;

import org.commonmark.node.Link;

public class ChatBeanAdapter extends BaseAdapter {
    private static final int TYPE_LEFT = 0;
    private static final int TYPE_RIGHT = 1;
    private static final int TYPE_COUNT = 2;
    private boolean mEnableMarkdown;
    private Context mContext;
    private ArrayList<ChatBean> mData;
    private LayoutInflater mLayoutInflater;
    private Markwon mMarkwon;

    public ChatBeanAdapter() {
    }

    public ChatBeanAdapter(Context context, ArrayList<ChatBean> data, boolean enableMarkdown) {
        mContext = context;
        mEnableMarkdown = enableMarkdown;
        if (mEnableMarkdown) {
            mMarkwon = Markwon.builder(context)
                    .usePlugin(new AbstractMarkwonPlugin() {
                        @Override
                        public void configureSpansFactory(
                                @NonNull MarkwonSpansFactory.Builder builder) {
                            builder.setFactory(Link.class, (configuration, props) -> {
                                String url = CoreProps.LINK_DESTINATION.get(props);
                                return new URLSpan(url) {
                                    @Override
                                    public void onClick(View widget) {
                                        if (!TextUtils.isEmpty(url)) {
                                            handleLinkClick(url);
                                        }
                                    }
                                };
                            });
                        }
                    }).build();
        }

        mLayoutInflater = LayoutInflater.from(context);
        this.mData = data;
    }


    private void handleLinkClick(String url) {
        if (url.startsWith("http") || url.startsWith("https")) {
            // 启动 WebViewActivity 并传递 URL
            Intent intent = new Intent(mContext, WebViewActivity.class);
            intent.putExtra("URL", url);
            mContext.startActivity(intent);
        } else {
            // 处理自定义跳转
            Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
            mContext.startActivity(intent);
        }
    }

    @Override
    public int getCount() {
        return mData != null ? mData.size() : 0;
    }

    @Override
    public Object getItem(int position) {
        return mData.get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public int getViewTypeCount() {
        return TYPE_COUNT;
    }

    @Override
    public int getItemViewType(int pos) {
        ChatBean chat = (ChatBean) mData.get(pos);
        if (chat.isMe) {
            return TYPE_RIGHT;
        }
        return TYPE_LEFT;
    }

    @Override
    public View getView(int position, View convertView, ViewGroup parent) {
       int type = getItemViewType(position);
       ViewHolder holder = null;
       if (convertView == null) {
           holder = new ViewHolder();
           if (type == TYPE_LEFT) {
                convertView = mLayoutInflater.inflate(R.layout.chat_left, null);
                holder.layoutRes = R.layout.chat_left;
                holder.say = convertView.findViewById(R.id.say_content);
                holder.time = convertView.findViewById(R.id.say_time);
                convertView.setTag(holder);
           } else if (type == TYPE_RIGHT) {
               convertView = mLayoutInflater.inflate(R.layout.chat_right, null);
               holder.layoutRes = R.layout.chat_right;
               holder.img = convertView.findViewById(R.id.chat_image);
               holder.say = convertView.findViewById(R.id.say_content);
               holder.time = convertView.findViewById(R.id.say_time);
               convertView.setTag(holder);
           }
       } else {
           if (type == TYPE_LEFT) {
               holder = (ViewHolder) convertView.getTag();
           } else if (type == TYPE_RIGHT) {
               holder = (ViewHolder) convertView.getTag();
           }
       }

        ChatBean chat = (ChatBean) mData.get(position);
        if (holder != null) {
            if (mEnableMarkdown) {
                mMarkwon.setMarkdown((TextView) holder.say, chat.say);
            } else {
                holder.say.setText(chat.say);
            }
            holder.time.setText(chat.time);

            if (holder.img != null) {
                if (TextUtils.isEmpty(chat.imagePath)) {
                    holder.img.setVisibility(View.GONE);
                } else {
                    holder.img.setVisibility(View.VISIBLE);
                    holder.img.setImageURI(Uri.fromFile(new File(chat.imagePath)));
                }
            }
        }
        return convertView;
    }

    public static class ViewHolder {
        ImageView img;
        TextView say;
        TextView time;
        int layoutRes;
    }

}

